<?php
/**
 * Plugin Name: BOFH Excuse of the Day
 * Description: Manage BOFH excuses as a custom post type. REST API to add excuses, shortcode [bofh_excuse] to display a random one.
 * Version: 1.0
 * Author: Your Name
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class BOFH_Excuse_CPT {

    public static function init() {
        add_action( 'init', [ __CLASS__, 'register_cpt' ] );
        add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
        add_shortcode( 'bofh_excuse', [ __CLASS__, 'shortcode_random_excuse' ] );
    }

    /**
     * Register Custom Post Type for excuses
     */
    public static function register_cpt() {
        $labels = [
            'name'               => 'Excuses',
            'singular_name'      => 'Excuse',
            'menu_name'          => 'Excuses',
            'name_admin_bar'     => 'Excuse',
            'add_new'            => 'Add New',
            'add_new_item'       => 'Add New Excuse',
            'new_item'           => 'New Excuse',
            'edit_item'          => 'Edit Excuse',
            'view_item'          => 'View Excuse',
            'all_items'          => 'All Excuses',
            'search_items'       => 'Search Excuses',
            'not_found'          => 'No excuses found.',
            'not_found_in_trash' => 'No excuses found in Trash.',
        ];

        $args = [
            'labels'             => $labels,
            'public'             => false,
            'show_ui'            => true,
            'show_in_menu'       => true,
            'supports'           => [ 'title', 'editor' ],
            'show_in_rest'       => false, // Gutenberg + API
        ];

        register_post_type( 'bofh_excuse', $args );
    }

    /**
     * Register custom REST endpoint to add excuses
     */
    public static function register_routes() {
        register_rest_route( 'bofh-excuse/v1', '/add', [
            'methods'             => 'POST',
            'callback'            => [ __CLASS__, 'api_add_excuse' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args'                => [
                'excuse' => [
                    'required' => true,
                    'type'     => 'string',
                ],
            ],
        ] );
    }

    /**
     * REST API callback: Add a new excuse
     */
    public static function api_add_excuse( $request ) {
        $excuse = sanitize_text_field( $request->get_param( 'excuse' ) );

        if ( empty( $excuse ) ) {
            return new WP_Error( 'no_excuse', 'Excuse cannot be empty', [ 'status' => 400 ] );
        }

        $post_id = wp_insert_post( [
            'post_type'   => 'bofh_excuse',
            'post_title'  => wp_trim_words( $excuse, 8, '…' ),
            'post_content'=> $excuse,
            'post_status' => 'publish',
        ] );

        if ( is_wp_error( $post_id ) ) {
            return $post_id;
        }

        return [ 'success' => true, 'id' => $post_id, 'excuse' => $excuse ];
    }

    /**
     * Shortcode handler: display a random excuse
     */
    public static function shortcode_random_excuse( $atts ) {
        $query = new WP_Query( [
            'post_type'      => 'bofh_excuse',
            'posts_per_page' => 1,
            'orderby'        => 'rand',
        ] );

        if ( ! $query->have_posts() ) {
            return '<p>No excuses available.</p>';
        }

        $query->the_post();
        $output = '<div class="bofh-excuse">' . esc_html( get_the_content() ) . '</div>';
        wp_reset_postdata();

        return $output;
    }
}

BOFH_Excuse_CPT::init();
