// clock.js — bootstrap to process queued inits
(function () {
  function runInit(item) {
    try {
      if (typeof window.initializeClocks === 'function') {
        window.initializeClocks(item.varName, item.instanceId);
      } else {
        console.warn('[clock] initializeClocks() not found');
      }
    } catch (e) {
      console.error('[clock] initializeClocks threw', e);
    }
  }

  function flushQueue() {
    var q = window.clockInitQueue || [];
    while (q.length) runInit(q.shift());
  }

  // Process anything already queued once DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', flushQueue);
  } else {
    flushQueue();
  }

  // If more items are pushed later, wrap push to auto-run
  window.clockInitQueue = window.clockInitQueue || [];
  var origPush = window.clockInitQueue.push;
  window.clockInitQueue.push = function () {
    for (var i = 0; i < arguments.length; i++) runInit(arguments[i]);
    return origPush.apply(window.clockInitQueue, arguments);
  };
})();


function initializeClocks(clockDataVar, instanceId) {
    const clockData = window[clockDataVar];

    // Extract locations & settings
    const locations = (clockData && Array.isArray(clockData.locations))
        ? clockData.locations
        : (Array.isArray(clockData) ? clockData : []);
    const settings = (clockData && clockData.settings) ? clockData.settings : {
        background: "#ffffff",
        hands: "#000000",
        seconds: "#ff0000",
        digital: "#000000",
        timezone: "#000000",
        border: "#000000",
        numbers: "#000000"
    };

    function drawClock(canvasId, digitalId, labelId, timeZone) {
        const canvas = document.getElementById(canvasId);
        const digital = document.getElementById(digitalId);
        const label = document.getElementById(labelId);
        if (!canvas) return;

        const ctx = canvas.getContext("2d");
        const width = canvas.width;
        const height = canvas.height;
        const radius = Math.min(width, height) / 2;
        const clockRadius = radius * 0.9;

        // Apply colors to labels
        if (digital) {
            digital.style.color = settings.digital || "#000";
            digital.style.setProperty("color", settings.digital || "#000", "important");
        }
        if (label) {
            label.style.color = settings.timezone || "#000";
            label.style.setProperty("color", settings.timezone || "#000", "important");
        }

        function drawFace(ctx) {
            ctx.beginPath();
            ctx.arc(0, 0, clockRadius, 0, Math.PI * 2);
            ctx.fillStyle = settings.background;
            ctx.fill();

            ctx.lineWidth = 5;
            ctx.strokeStyle = settings.border;
            ctx.stroke();

            // center dot
            ctx.beginPath();
            ctx.arc(0, 0, 5, 0, Math.PI * 2);
            ctx.fillStyle = settings.hands;
            ctx.fill();
        }

        function drawNumbers(ctx) {
            ctx.fillStyle = settings.numbers || settings.hands;
            ctx.font = `${clockRadius * 0.15}px Arial`;
            ctx.textBaseline = "middle";
            ctx.textAlign = "center";

            for (let num = 1; num <= 12; num++) {
                const ang = (num * Math.PI) / 6;
                ctx.save();
                ctx.rotate(ang);
                ctx.translate(0, -clockRadius * 0.85);
                ctx.rotate(-ang);
                ctx.fillText(String(num), 0, 0);
                ctx.restore();
            }
        }

        function drawHands(ctx, localTime) {
            const hour = localTime.getHours();
            const minute = localTime.getMinutes();
            const second = localTime.getSeconds();

            const hourAngle = ((hour % 12) + minute / 60) * Math.PI / 6;
            const minuteAngle = (minute + second / 60) * Math.PI / 30;
            const secondAngle = second * Math.PI / 30;

            // Hour
            ctx.save();
            ctx.rotate(hourAngle);
            ctx.beginPath();
            ctx.lineWidth = 8;
            ctx.lineCap = "round";
            ctx.strokeStyle = settings.hands;
            ctx.moveTo(0, 0);
            ctx.lineTo(0, -clockRadius * 0.5);
            ctx.stroke();
            ctx.restore();

            // Minute
            ctx.save();
            ctx.rotate(minuteAngle);
            ctx.beginPath();
            ctx.lineWidth = 6;
            ctx.lineCap = "round";
            ctx.strokeStyle = settings.hands;
            ctx.moveTo(0, 0);
            ctx.lineTo(0, -clockRadius * 0.7);
            ctx.stroke();
            ctx.restore();

            // Second
            ctx.save();
            ctx.rotate(secondAngle);
            ctx.beginPath();
            ctx.lineWidth = 2;
            ctx.lineCap = "round";
            ctx.strokeStyle = settings.seconds;
            ctx.moveTo(0, 0);
            ctx.lineTo(0, -clockRadius * 0.9);
            ctx.stroke();
            ctx.restore();
        }

        function update() {
            ctx.setTransform(1, 0, 0, 1, 0, 0);
            ctx.clearRect(0, 0, width, height);
            ctx.translate(width / 2, height / 2);

            drawFace(ctx);
            drawNumbers(ctx);

            let localTime;
            try {
                localTime = new Date(new Date().toLocaleString("en-US", { timeZone }));
            } catch {
                localTime = new Date();
            }

            if (digital) {
                const h = String(localTime.getHours()).padStart(2, "0");
                const m = String(localTime.getMinutes()).padStart(2, "0");
                const s = String(localTime.getSeconds()).padStart(2, "0");
                digital.textContent = `${h}:${m}:${s}`;
            }

            drawHands(ctx, localTime);

            ctx.setTransform(1, 0, 0, 1, 0, 0);
        }

        update();
        setInterval(update, 1000);
    }

    locations.forEach((location) => {
        const id = location.toLowerCase().replace(/[/\s]/g, "-") + "-" + instanceId;
        drawClock(`${id}-clock`, `${id}-digital`, `${id}-label`, location);
    });
}

document.addEventListener("DOMContentLoaded", function() {
  if (window.clockInitQueue) {
    window.clockInitQueue.forEach(function(entry) {
      if (typeof window.initializeClocks === "function") {
        window.initializeClocks(entry.varName, entry.instanceId);
      }
    });
  }
});
