<?php
/**
 * Plugin Name: Cookie Consent
 * Plugin URI: https://it-breeze.cloud
 * Update URI: https://it-breeze.cloud/data/cookie-consent/cookie-consent.json
 * Description: A simple, customizable cookie consent banner for WordPress
 * Version: 2.2.0
 * Author: Mike Vahldieck
 * License: GPL v2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

class CustomCookieConsent {
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'display_cookie_banner'));
        add_action('wp_ajax_accept_cookies', array($this, 'handle_accept_cookies'));
        add_action('wp_ajax_nopriv_accept_cookies', array($this, 'handle_accept_cookies'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('cookie-consent-js', plugin_dir_url(__FILE__) . 'cookie-consent.js', array('jquery'), '1.0.0', true);
        wp_enqueue_style('cookie-consent-css', plugin_dir_url(__FILE__) . 'cookie-consent.css', array(), '1.0.0');
        wp_localize_script('cookie-consent-js', 'cookieConsent', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cookie_consent_nonce')
        ));
    }
    
    public function display_cookie_banner() {
        if (isset($_COOKIE['cookie_consent']) && $_COOKIE['cookie_consent'] === 'accepted') {
            return;
        }

        $message = get_option('cookie_consent_message', 'This website uses cookies to ensure you get the best experience on our website.');
        $accept_text = get_option('cookie_consent_accept_text', 'Accept');
        $decline_text = get_option('cookie_consent_decline_text', 'Decline');
        $learn_more_text = get_option('cookie_consent_learn_more_text', 'Learn More');
        $privacy_url = get_option('cookie_consent_privacy_url', '');
        $image_url = get_option('cookie_consent_image_url', '');
        $position = get_option('cookie_consent_position', 'bottom');
        $show_decline = get_option('cookie_consent_show_decline', '1');
        $bg_color = get_option('cookie_consent_bg_color', '#222222');
        $text_color = get_option('cookie_consent_text_color', '#ffffff');
        $btn_color = get_option('cookie_consent_button_color', '#0073aa');
        $btn_hover_color = get_option('cookie_consent_button_hover_color', '#005177');
        ?>
        <style>
            #cookie-consent-banner {
                background-color: <?php echo esc_html($bg_color); ?>;
                color: <?php echo esc_html($text_color); ?>;
            }
            #cookie-consent-banner .cookie-consent-btn {
                background-color: <?php echo esc_html($btn_color); ?>;
                color: #fff;
                border: none;
                border-radius: 3px;
                padding: 8px 15px;
                cursor: pointer;
            }
            #cookie-consent-banner .cookie-consent-btn:hover {
                background-color: <?php echo esc_html($btn_hover_color); ?>;
            }
            #cookie-consent-banner a {
                color: <?php echo esc_html($text_color); ?>;
                text-decoration: underline;
            }
        </style>

        <div id="cookie-consent-banner" class="cookie-consent-banner cookie-consent-<?php echo esc_attr($position); ?>" style="display: none;">
            <div class="cookie-consent-content">
                <div class="cookie-consent-text-section">
                    <?php if ($image_url): ?>
                        <div class="cookie-consent-image">
                            <img src="<?php echo esc_url($image_url); ?>" alt="Cookie consent" />
                        </div>
                    <?php endif; ?>
                    <div class="cookie-consent-message">
                        <?php echo esc_html($message); ?>
                        <?php if (!empty($privacy_url)): ?>
                            <a href="<?php echo esc_url($privacy_url); ?>" target="_blank" class="cookie-consent-learn-more">
                                <?php echo esc_html($learn_more_text); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="cookie-consent-buttons">
                    <button id="cookie-consent-accept" class="cookie-consent-btn cookie-consent-accept">
                        <?php echo esc_html($accept_text); ?>
                    </button>
                    <?php if ($show_decline === '1'): ?>
                        <button id="cookie-consent-decline" class="cookie-consent-btn cookie-consent-decline">
                            <?php echo esc_html($decline_text); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }

    public function handle_accept_cookies() {
        if (!wp_verify_nonce($_POST['nonce'], 'cookie_consent_nonce')) {
            wp_die('Security check failed');
        }
        $action = sanitize_text_field($_POST['action_type']);
        if ($action === 'accept') {
            setcookie('cookie_consent', 'accepted', time() + (365 * 24 * 60 * 60), '/');
        } elseif ($action === 'decline') {
            setcookie('cookie_consent', 'declined', time() + (365 * 24 * 60 * 60), '/');
        }
        wp_send_json_success();
    }
    
    public function add_admin_menu() {
        add_options_page(
            'Cookie Consent Settings',
            'Cookie Consent',
            'manage_options',
            'cookie-consent-settings',
            array($this, 'admin_page')
        );
    }
    
    public function register_settings() {
        register_setting('cookie_consent_settings', 'cookie_consent_message');
        register_setting('cookie_consent_settings', 'cookie_consent_accept_text');
        register_setting('cookie_consent_settings', 'cookie_consent_decline_text');
        register_setting('cookie_consent_settings', 'cookie_consent_learn_more_text');
        register_setting('cookie_consent_settings', 'cookie_consent_privacy_url');
        register_setting('cookie_consent_settings', 'cookie_consent_image_url');
        register_setting('cookie_consent_settings', 'cookie_consent_position');
        register_setting('cookie_consent_settings', 'cookie_consent_show_decline');
        register_setting('cookie_consent_settings', 'cookie_consent_show_image');
        register_setting('cookie_consent_settings', 'cookie_consent_bg_color');
        register_setting('cookie_consent_settings', 'cookie_consent_text_color');
        register_setting('cookie_consent_settings', 'cookie_consent_button_color');
        register_setting('cookie_consent_settings', 'cookie_consent_button_hover_color');
    }

    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>Cookie Consent Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('cookie_consent_settings'); ?>
                <?php do_settings_sections('cookie_consent_settings'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">Cookie Message</th>
                        <td>
                            <textarea name="cookie_consent_message" rows="3" cols="50"><?php echo esc_textarea(get_option('cookie_consent_message', 'This website uses cookies to ensure you get the best experience on our website.')); ?></textarea>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Accept Button Text</th>
                        <td>
                            <input type="text" name="cookie_consent_accept_text" value="<?php echo esc_attr(get_option('cookie_consent_accept_text', 'Accept')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Decline Button Text</th>
                        <td>
                            <input type="text" name="cookie_consent_decline_text" value="<?php echo esc_attr(get_option('cookie_consent_decline_text', 'Decline')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Learn More Text</th>
                        <td>
                            <input type="text" name="cookie_consent_learn_more_text" value="<?php echo esc_attr(get_option('cookie_consent_learn_more_text', 'Learn More')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Privacy Policy URL</th>
                        <td>
                            <input type="url" name="cookie_consent_privacy_url" value="<?php echo esc_url(get_option('cookie_consent_privacy_url', '')); ?>" style="width: 400px;" />
                            <p class="description">Link to your privacy policy page (optional)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Show Image</th>
                        <td>
                            <input type="checkbox" name="cookie_consent_show_image" value="1" <?php checked(get_option('cookie_consent_show_image', '1'), '1'); ?> />
                            <label>Display cookie image next to the message</label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Cookie Image</th>
                        <td>
                            <?php
                            $image_url = get_option('cookie_consent_image_url', '');
                            $show_image = get_option('cookie_consent_show_image', '1');
                            $default_image = plugin_dir_url(__FILE__) . 'default-cookie.png';
                            $preview_image = $image_url ? esc_url($image_url) : esc_url($default_image);
                            ?>
                            <img id="cookie-consent-image-preview" src="<?php echo $preview_image; ?>" style="max-width:100px; display:block; margin-bottom:10px;">
                            <input type="hidden" id="cookie_consent_image_url" name="cookie_consent_image_url" value="<?php echo esc_url($image_url); ?>" />
                            <button type="button" class="button" id="upload_cookie_image_button">Select Image</button>
                            <button type="button" class="button" id="remove_cookie_image_button">Remove</button>
                            <p class="description">Choose an image from the media library. If none is selected, the default image will be used.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Banner Position</th>
                        <td>
                            <select name="cookie_consent_position">
                                <option value="bottom" <?php selected(get_option('cookie_consent_position', 'bottom'), 'bottom'); ?>>Bottom</option>
                                <option value="top" <?php selected(get_option('cookie_consent_position', 'bottom'), 'top'); ?>>Top</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Show Decline Button</th>
                        <td>
                            <input type="checkbox" name="cookie_consent_show_decline" value="1" <?php checked(get_option('cookie_consent_show_decline', '1'), '1'); ?> />
                            <label>Show decline button</label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Background Color</th>
                        <td>
                            <input type="color" name="cookie_consent_bg_color" value="<?php echo esc_attr(get_option('cookie_consent_bg_color', '#222222')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Text Color</th>
                        <td>
                            <input type="color" name="cookie_consent_text_color" value="<?php echo esc_attr(get_option('cookie_consent_text_color', '#ffffff')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Button Color</th>
                        <td>
                            <input type="color" name="cookie_consent_button_color" value="<?php echo esc_attr(get_option('cookie_consent_button_color', '#0073aa')); ?>" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Button Hover Color</th>
                        <td>
                            <input type="color" name="cookie_consent_button_hover_color" value="<?php echo esc_attr(get_option('cookie_consent_button_hover_color', '#005177')); ?>" />
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>

        <script>
        jQuery(document).ready(function($) {
            var mediaUploader;
            $('#upload_cookie_image_button').on('click', function(e) {
                e.preventDefault();
                if (mediaUploader) {
                    mediaUploader.open();
                    return;
                }
                mediaUploader = wp.media.frames.file_frame = wp.media({
                    title: 'Select Cookie Image',
                    button: { text: 'Select Image' },
                    multiple: false
                });
                mediaUploader.on('select', function() {
                    var attachment = mediaUploader.state().get('selection').first().toJSON();
                    $('#cookie_consent_image_url').val(attachment.url);
                    $('#cookie-consent-image-preview').attr('src', attachment.url);
                });
                mediaUploader.open();
            });

            $('#remove_cookie_image_button').on('click', function() {
                $('#cookie_consent_image_url').val('');
                $('#cookie-consent-image-preview').attr('src', '<?php echo esc_js($default_image); ?>');
            });
        });
        </script>
        <?php
    }
}

new CustomCookieConsent();

register_activation_hook(__FILE__, 'cookie_consent_activate');
function cookie_consent_activate() {
    add_option('cookie_consent_message', 'This website uses cookies to ensure you get the best experience on our website.');
    add_option('cookie_consent_accept_text', 'Accept');
    add_option('cookie_consent_decline_text', 'Decline');
    add_option('cookie_consent_learn_more_text', 'Learn More');
    add_option('cookie_consent_position', 'bottom');
    add_option('cookie_consent_show_decline', '1');
}

register_deactivation_hook(__FILE__, 'cookie_consent_deactivate');
function cookie_consent_deactivate() {
    delete_option('cookie_consent_message');
    delete_option('cookie_consent_accept_text');
    delete_option('cookie_consent_decline_text');
    delete_option('cookie_consent_learn_more_text');
    delete_option('cookie_consent_privacy_url');
    delete_option('cookie_consent_position');
    delete_option('cookie_consent_show_decline');
}

require_once plugin_dir_path(__FILE__) . 'update.php';
new cookie_consent_Plugin_Updater(__FILE__, 'cookie_consent', 'https://it-breeze.cloud/data/cookie-consent/cookie-consent.json');
