<?php
/**
 * Plugin Name: Debug
 * Plugin URI: https://it-breeze.info/
 * Update URI: https://it-breeze.cloud/data/debug/debug.json
 * Description: calculate and display a Debug consent banner.
 * Version: 2.2.0
 * Author: Mike Vahldieck
 * Author URI: https://it-breeze.info/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Add admin bar toggle
 */
add_action( 'admin_bar_menu', function( $wp_admin_bar ) {
    if ( ! current_user_can( 'manage_options' ) ) return;

    $debug_status = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? 'ON' : 'OFF';
    $toggle_label = ( WP_DEBUG ) ? 'Disable Debug' : 'Enable Debug';

    $wp_admin_bar->add_node( [
        'id'    => 'debug_mode_toggle',
        'title' => 'Debug: ' . $debug_status,
        'href'  => '#',
        'meta'  => [ 'title' => $toggle_label ]
    ] );
}, 100 );

/**
 * Enqueue JS for AJAX toggle
 */
add_action( 'admin_enqueue_scripts', function() {
    if ( ! is_admin_bar_showing() ) return;
    wp_enqueue_script( 'debug-toggle', plugin_dir_url(__FILE__) . 'toggle-debug.js', ['jquery'], '2.1.0', true );
    wp_localize_script( 'debug-toggle', 'debugModeData', [
        'ajaxurl' => admin_url( 'admin-ajax.php' ),
        'nonce'   => wp_create_nonce( 'toggle_debug_mode' ),
    ] );
});

/**
 * AJAX handler for toggling debug mode
 */
add_action( 'wp_ajax_toggle_debug_mode', 'toggle_debug_mode_ajax' );

function toggle_debug_mode_ajax() {
    check_ajax_referer( 'toggle_debug_mode', 'nonce' );

    $result = toggle_debug_mode_direct();

    if ( is_wp_error( $result ) ) {
        wp_send_json_error( $result->get_error_message() );
    } else {
        wp_send_json_success( [ 'status' => $result ] );
    }
}

function toggle_debug_mode_direct() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return new WP_Error( 'unauthorized', 'You are not allowed to do this.' );
    }

    $wp_config_path = ABSPATH . 'wp-config.php';

    if ( ! file_exists( $wp_config_path ) || ! is_writable( $wp_config_path ) ) {
        return new WP_Error( 'config_error', 'wp-config.php is missing or not writable.' );
    }

    $config = file_get_contents( $wp_config_path );

    if ( preg_match( "/define\s*\(\s*'WP_DEBUG'\s*,\s*true\s*\)\s*;/", $config ) ) {
        $config = preg_replace( "/define\s*\(\s*'WP_DEBUG'\s*,\s*true\s*\)\s*;/", "define('WP_DEBUG', false);", $config );
        $status = 'OFF';
    } elseif ( preg_match( "/define\s*\(\s*'WP_DEBUG'\s*,\s*false\s*\)\s*;/", $config ) ) {
        $config = preg_replace( "/define\s*\(\s*'WP_DEBUG'\s*,\s*false\s*\)\s*;/", "define('WP_DEBUG', true);", $config );
        $status = 'ON';
    } else {
        // Add define if missing
        $config = preg_replace( "/(<\?php)/", "$1\ndefine('WP_DEBUG', true);", $config, 1 );
        $status = 'ON';
    }

    file_put_contents( $wp_config_path, $config );

    return $status;
}


/**
 * Add settings page
 */
add_action( 'admin_menu', function() {
    add_options_page(
        'Debug Mode Settings',
        'Debug Mode',
        'manage_options',
        'debug-mode',
        'debug_mode_settings_page'
    );
});

function debug_mode_settings_page() {
    $message = '';

    if ( isset( $_POST['toggle_debug_nonce'] ) && wp_verify_nonce( $_POST['toggle_debug_nonce'], 'toggle_debug_mode' ) ) {
        $result = toggle_debug_mode_direct();
        if ( is_wp_error( $result ) ) {
            $message = '<div class="notice notice-error"><p>' . esc_html( $result->get_error_message() ) . '</p></div>';
        } else {
            // Show message briefly, then reload
            echo '<div class="notice notice-success"><p>Debug mode toggled. It is now <strong>' . esc_html( $result ) . '</strong>.</p></div>';
            echo '<meta http-equiv="refresh" content="1">';
        }
    }

    $debug_status = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? 'ON' : 'OFF';
    ?>
    <div class="wrap">
        <h1>Debug Mode Settings</h1>
        <?php echo $message; ?>
        <p>Current status: <strong><?php echo esc_html( $debug_status ); ?></strong></p>
        <form method="post">
            <?php wp_nonce_field( 'toggle_debug_mode', 'toggle_debug_nonce' ); ?>
            <input type="submit" class="button button-primary" value="Toggle Debug Mode">
        </form>
    </div>
    <?php
}

require_once plugin_dir_path(__FILE__) . 'update.php';
new debug_Plugin_Updater(__FILE__, 'debug', 'https://it-breeze.cloud/data/debug/debug.json');
