<?php
/**
 * Plugin Name: Maintenance mode
 * Plugin URI: https://it-breeze.info
 * Update URI: https://it-breeze.cloud/data/maintenance/maintenance.json
 * Description:  A plugin to put the site into maintenance mode.
 * Version: 1.2.5
 * Author: Mike Vahldieck
 * Author URI: https://it-breeze.info
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

define('MAINTENANCE_DEFAULT_BG_URL', plugins_url('background.png', __FILE__));
define('MAINTENANCE_DEFAULT_MESSAGE', "🚧 Site Under Construction 🚧\nDon't worry, we haven't been abducted by aliens... we're just giving our website a spa day. It should be back looking fabulous shortly!");

add_filter('default_option_maintenance_mode_background_image', function ($default) {
    return MAINTENANCE_DEFAULT_BG_URL;
});

add_filter('default_option_maintenance_mode_message', function () {
    return MAINTENANCE_DEFAULT_MESSAGE;
});

add_action('admin_menu', 'maintenance_mode_menu');
function maintenance_mode_menu() {
    add_options_page('Maintenance Mode Settings', 'Maintenance Mode', 'manage_options', 'maintenance-mode-plugin', 'maintenance_mode_settings_page');
}

add_action('admin_init', 'maintenance_mode_settings');
function maintenance_mode_settings() {
    register_setting('maintenance-mode-settings-group', 'maintenance_mode');
    register_setting('maintenance-mode-settings-group', 'maintenance_mode_background_image');
    register_setting('maintenance-mode-settings-group', 'maintenance_mode_message');
    register_setting('maintenance-mode-settings-group', 'maintenance_mode_message_position');
    register_setting('maintenance-mode-settings-group', 'maintenance_mode_message_image');
}

add_action('admin_enqueue_scripts', 'maintenance_mode_enqueue_scripts');
function maintenance_mode_enqueue_scripts($hook) {
    if ($hook != 'settings_page_maintenance-mode-plugin') {
        return;
    }
    wp_enqueue_media();
    wp_enqueue_script('fdm-tab-switcher', plugin_dir_url(__FILE__) . 'maintenance.js', array('jquery'), null, true);
}

add_action('admin_bar_menu', 'maintenance_mode_admin_bar_toggle', 100);
function maintenance_mode_admin_bar_toggle($wp_admin_bar) {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $status = get_option('maintenance_mode') ? 'ON' : 'OFF';
    $toggle_url = add_query_arg('toggle_maintenance_mode', '1', admin_url());
    
    $wp_admin_bar->add_menu(array(
        'id'    => 'maintenance_mode_status',
        'title' => 'Maintenance: ' . $status,
        'href'  => $toggle_url
    ));
}

add_action('admin_init', 'toggle_maintenance_mode');
function toggle_maintenance_mode() {
    if (isset($_GET['toggle_maintenance_mode']) && current_user_can('manage_options')) {
        $current_status = get_option('maintenance_mode');
        update_option('maintenance_mode', !$current_status);
        $redirect_url = remove_query_arg('toggle_maintenance_mode', wp_get_referer());
        wp_safe_redirect($redirect_url);
        exit;
    }
}

function maintenance_mode_settings_page() {
    ?>
    <div class="wrap">
        <h1>Maintenance Mode Settings</h1>
        <form method="post" action="options.php">
            <?php settings_fields('maintenance-mode-settings-group'); ?>
            <?php do_settings_sections('maintenance-mode-settings-group'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Enable Maintenance Mode</th>
                    <td><input type="checkbox" name="maintenance_mode" value="1" <?php checked(1, get_option('maintenance_mode'), true); ?>></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Background Image</th>
                    <td>
                        <input type="hidden" name="maintenance_mode_background_image" id="maintenance_mode_background_image" value="<?php echo esc_attr(get_option('maintenance_mode_background_image')); ?>" />
                        <div id="maintenance_mode_background_image_preview">
                            
                            <?php 
                                $background_image = get_option('maintenance_mode_background_image');
                                if (empty($background_image)) {
                                    $background_image = MAINTENANCE_DEFAULT_BG_URL;
                                    echo '<img src="<?php echo esc_url($background_image); ?>" style="max-width: 100px; margin: 5px;">';
                                } 
                            ?>
                                

                        </div>
                        <input type="button" class="button" id="maintenance_mode_background_image_button" value="Upload Image" />
                        <input type="button" class="button" id="maintenance_mode_background_image_remove" value="Remove Image" style="display: <?php echo $background_image ? 'inline-block' : 'none'; ?>;">
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Maintenance Message</th>    
                    <?php 
                    $message = get_option('maintenance_mode_message');
                    if (empty($message)) {
                        $message = MAINTENANCE_DEFAULT_MESSAGE;
                    }
                    ?>
                    <td>
                        <textarea name="maintenance_mode_message" rows="5" cols="50"><?php echo esc_textarea($message); ?></textarea>
                    </td>
                    
                </tr>
                <tr valign="top">
                    <th scope="row">Message Position</th>
                    <td>
                        <select name="maintenance_mode_message_position">
                            <option value="top-left" <?php selected('top-left', get_option('maintenance_mode_message_position')); ?>>Top Left</option>
                            <option value="top-center" <?php selected('top-center', get_option('maintenance_mode_message_position')); ?>>Top Center</option>
                            <option value="top-right" <?php selected('top-right', get_option('maintenance_mode_message_position')); ?>>Top Right</option>
                            <option value="center-left" <?php selected('center-left', get_option('maintenance_mode_message_position')); ?>>Center Left</option>
                            <option value="center" <?php selected('center', get_option('maintenance_mode_message_position')); ?>>Center</option>
                            <option value="center-right" <?php selected('center-right', get_option('maintenance_mode_message_position')); ?>>Center Right</option>
                            <option value="bottom-left" <?php selected('bottom-left', get_option('maintenance_mode_message_position')); ?>>Bottom Left</option>
                            <option value="bottom-center" <?php selected('bottom-center', get_option('maintenance_mode_message_position')); ?>>Bottom Center</option>
                            <option value="bottom-right" <?php selected('bottom-right', get_option('maintenance_mode_message_position')); ?>>Bottom Right</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Message Image</th>
                    <td>
                        <input type="hidden" name="maintenance_mode_message_image" id="maintenance_mode_message_image" value="<?php echo esc_attr(get_option('maintenance_mode_message_image')); ?>" />
                        <div id="maintenance_mode_message_image_preview">
                            <?php if ($message_image = get_option('maintenance_mode_message_image')): ?>
                                <img src="<?php echo esc_url($message_image); ?>" style="max-width: 100px; margin: 5px;">
                            <?php endif; ?>
                        </div>
                        <input type="button" class="button" id="maintenance_mode_message_image_button" value="Upload Image" />
                        <input type="button" class="button" id="maintenance_mode_message_image_remove" value="Remove Image" style="display: <?php echo $message_image ? 'inline-block' : 'none'; ?>;">
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}
add_action('template_redirect', 'maintenance_mode_page');
function maintenance_mode_page() {
    if (!is_admin() && !current_user_can('manage_options') && get_option('maintenance_mode')) {
        $background_image = get_option('maintenance_mode_background_image');
        if (empty($background_image)) {
            $background_image = MAINTENANCE_DEFAULT_BG_URL;
        }
        $background_image = esc_url($background_image);
        $login_url = wp_login_url();
        $message_image = esc_url(get_option('maintenance_mode_message_image'));
        $maintenance_message = get_option('maintenance_mode_message');
        if (empty($maintenance_message)) {
            $maintenance_message = MAINTENANCE_DEFAULT_MESSAGE;
        }
        $maintenance_message = wp_kses(nl2br($maintenance_message), array('br' => array()));
        $message_position = esc_attr(get_option('maintenance_mode_message_position'));
        $site_name = get_bloginfo('name');
        $position_css = array(
            'top-left' => 'top: 10px; left: 10px;',
            'top-center' => 'top: 10px; left: 50%; transform: translateX(-50%);',
            'top-right' => 'top: 10px; right: 10px;',
            'center-left' => 'top: 50%; left: 10px; transform: translateY(-50%);',
            'center' => 'top: 50%; left: 50%; transform: translate(-50%, -50%);',
            'center-right' => 'top: 50%; right: 10px; transform: translateY(-50%);',
            'bottom-left' => 'bottom: 10px; left: 10px;',
            'bottom-center' => 'bottom: 10px; left: 50%; transform: translateX(-50%);',
            'bottom-right' => 'bottom: 10px; right: 10px;',
        );
        $message_css = isset($position_css[$message_position]) ? $position_css[$message_position] : $position_css['center'];

        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title><?php echo esc_html($site_name); ?> - Maintenance Mode</title>
            <style>
                body {
                    text-align: center;
                    background: <?php echo $background_image ? "url('$background_image') no-repeat center center fixed" : '#f3f3f3'; ?>;
                    background-size: cover;
                    font-family: Arial, sans-serif;
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    align-items: center;
                    height: 100vh;
                    margin: 0;
                }
.maintenance-message {
    background: rgba(255, 255, 255, 1);
    color: #000;
    position: absolute;
    <?php echo $message_css; ?>
    padding: 20px;
    border-radius: 10px;
    z-index: 10;
    max-width: 400px; /* Add this line */
    width: auto; /* Add this line */
    box-sizing: border-box; /* Add this line */
}
                .message-image {
                    max-width: 150px;
                    margin-bottom: 15px;
                }
            </style>
        </head>
        <body>
            <div class="maintenance-message">
                <?php if ($message_image) { ?>
                    <img src="<?php echo esc_url($message_image); ?>" class="message-image" />
                <?php } ?>
                
                <p><?php echo $maintenance_message ? $maintenance_message : ""; ?></p>
                <p><a href='<?php echo $login_url; ?>' style='color: #126724; text-decoration: none;'>Log in</a></p>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

require_once plugin_dir_path(__FILE__) . 'update.php';
new maintenance_Plugin_Updater(__FILE__, 'maintenance', 'https://it-breeze.cloud/data/maintenance/maintenance.json');
