jQuery(document).ready(function($){
  const fileInput = $('#song-file');
  const progressBox = $('#upload-progress');
  const progressBar = $('#upload-bar');
  const status = $('#upload-status');
  const audioField = $('#audio_file');

  // Enable upload button if album info present; otherwise keep disabled and show hint
  function enableUploadIfAlbum() {
    const albumText = $('input[name="album"]').val() ? $('input[name="album"]').val().trim() : '';
    const albumId = $('#album_id').val() ? $('#album_id').val() : '';
    if (albumText || albumId) {
      fileInput.prop('disabled', false);
      $('#upload-disabled-hint').remove();
      return true;
    } else {
      fileInput.prop('disabled', true);
      if (!$('#upload-disabled-hint').length) {
        $('<span id="upload-disabled-hint" class="description" style="margin-left:8px;">Upload enabled after album is selected or entered.</span>').insertAfter(fileInput);
      }
      return false;
    }
  }

  // Initial check on load (for edit screen where album may already exist)
  enableUploadIfAlbum();

  // When user types an album name, re-check
  $(document).on('input', 'input[name="album"]', function(){
    enableUploadIfAlbum();
  });

  // If hidden album_id is changed programmatically and triggers change, re-check
  $(document).on('change', '#album_id', function(){
    enableUploadIfAlbum();
  });

  // Also handle clicks on album list items (some handlers set album_id without change event)
  $(document).on('click', '.album-item', function(){
    // small timeout to let other inline handlers set fields first
    setTimeout(function(){ enableUploadIfAlbum(); }, 10);
  });

  // Handle file selection (upload starts only if enabled)
  fileInput.on('change', function(){
    if (fileInput.prop('disabled')) {
      // safety: clear selection if somehow chosen while disabled
      fileInput.val('');
      return;
    }

    const file = this.files[0];
    if (!file) return;

    // Determine album (prefer text, then id)
    const albumText = $('input[name="album"]').val() ? $('input[name="album"]').val().trim() : '';
    const albumId = $('#album_id').val() ? $('#album_id').val() : '';
    const album = albumText || albumId;

    if (!album) {
      alert('⚠️ Please select or enter an album before uploading a song file.');
      fileInput.val('');
      return;
    }

    const formData = new FormData();
    formData.append('action', 'song_manager_async_upload');
    formData.append('song_file', file);
    formData.append('album', album);

    // include nonce if available
    if (typeof musicUploader !== 'undefined' && musicUploader.nonce) {
      formData.append('nonce', musicUploader.nonce);
    }

    progressBox.show();
    progressBar.width('0%').css('background', '#0073aa');
    status.text('Uploading…');

    $.ajax({
      url: musicUploader.ajaxurl,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      xhr: function() {
        const xhr = $.ajaxSettings.xhr();
        if (xhr.upload) {
          xhr.upload.addEventListener('progress', function(e){
            if (e.lengthComputable) {
              const percent = Math.round((e.loaded / e.total) * 100);
              progressBar.width(percent + '%');
            }
          }, false);
        }
        return xhr;
      },
      success: function(resp){
        if (resp.success) {
          progressBar.css('background', '#46b450').width('100%');
          status.html('Uploaded ✓ <a href="'+resp.data.url+'" target="_blank">'+file.name+'</a>');
          audioField.val(resp.data.url);
        } else {
          progressBar.css('background', '#dc3232').width('100%');
          status.text('Error: ' + (resp.data?.message || resp.data));
        }
      },
      error: function(xhr, statusText, err){
        progressBar.css('background', '#dc3232').width('100%');
        status.text('Upload failed: ' + err);
      }
    });
  });
});

document.addEventListener('DOMContentLoaded', async () => {
  // 1. Get the site URL and stored license key from WordPress
  const siteUrl = window.location.origin;
  const licenseKey = window.musicPluginLicenseKey || ''; // defined via PHP (see step 4)

  // 2. Wait for the external license script to load
  if (window.MusicPluginLicense) {
    const license = await window.MusicPluginLicense.check(licenseKey, siteUrl);
    console.log("License status:", license);

    // 3. Apply free limits if the plan is 'free'
    if (license.plan === 'free') {
      enforceFreeLimits(license.maxAlbums, license.maxSongs);
    }
  }
});

function enforceFreeLimits(maxAlbums, maxSongs) {
  const addAlbumBtn = document.querySelector('#add-album-btn');
  const addSongBtn = document.querySelector('#add-song-btn');

  // You can adjust the selectors to match your buttons/elements
  const albumCount = document.querySelectorAll('.album-item').length;
  const songCount = document.querySelectorAll('.song-item').length;

  if (albumCount >= maxAlbums && addAlbumBtn) {
    addAlbumBtn.disabled = true;
    alert(`Free version limit reached: maximum ${maxAlbums} albums.`);
  }

  if (songCount >= maxSongs && addSongBtn) {
    addSongBtn.disabled = true;
    alert(`Free version limit reached: maximum ${maxSongs} songs.`);
  }
}
