<?php
/**
 * Plugin Name: Music
 * Plugin URI: https://it-breeze.info/
 * Update URI: https://it-breeze.cloud/data/music/music.json
 * Description: Display songs in post / pages. Shortcode: [music].
 * Version: 2.3.0
 * Author: Mike Vahldieck (modified)
 * Author URI: https://it-breeze.info/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    //exit;
}

register_activation_hook(__FILE__, 'song_manager_activate');
register_deactivation_hook(__FILE__, 'song_manager_deactivate');

function song_manager_activate() {
    global $wpdb;
    
    $table_songs = $wpdb->prefix . 'songs';
    $table_albums = $wpdb->prefix . 'albums';
    
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql_songs = "CREATE TABLE $table_songs (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        title varchar(255) NOT NULL,
        slug varchar(255) NOT NULL,
        artist varchar(255) DEFAULT '',
        genre varchar(100) DEFAULT '',
        album text DEFAULT NULL,
        cover text DEFAULT NULL,
        album_id mediumint(9) DEFAULT NULL,
        track mediumint(9) DEFAULT NULL,
        year year DEFAULT NULL,
        language varchar(50) DEFAULT '',
        audio_file varchar(500) DEFAULT '',
        lyrics text DEFAULT '',
        description text DEFAULT '',
        download_link varchar(500) DEFAULT '',
        buy_link varchar(500) DEFAULT '',
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY slug (slug)
    ) $charset_collate;";
    
    $sql_albums = "CREATE TABLE $table_albums (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        title varchar(255) NOT NULL,
        artist varchar(255) DEFAULT '',
        cover varchar(500) DEFAULT '',
        genre varchar(100) DEFAULT '',
        year varchar(10) DEFAULT '',
        language varchar(50) DEFAULT '',
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_songs);
    dbDelta($sql_albums);

}

function song_manager_deactivate() {
    // No cleanup on deactivation by design
}

function song_manager_get_db_connection() {
    $use_wp_db = get_option('song_manager_use_wp_db', false);
    if ($use_wp_db) {
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASSWORD,
                array(
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                )
            );
            
            $stmt = $pdo->query("SHOW TABLES LIKE 'songs'");
            if ($stmt->rowCount() > 0) {
                // Ensure albums table exists
                $stmt2 = $pdo->query("SHOW TABLES LIKE 'albums'");
                if ($stmt2->rowCount() == 0) {
                    song_manager_create_albums_table($pdo);
                }
                return $pdo;
            } else {
                song_manager_create_songs_table($pdo);
                song_manager_create_albums_table($pdo);
                return $pdo;
            }
        } catch(PDOException $e) {
            error_log("Song Manager WordPress database connection failed: " . $e->getMessage());
        }
    }

    $manual_host = get_option('song_manager_db_host');
    $manual_username = get_option('song_manager_db_username');
    $manual_database = get_option('song_manager_db_name');
    
    if ($manual_host && $manual_username && $manual_database) {
        try {
            $host = $manual_host;
            $port = get_option('song_manager_db_port', '3306');
            $username = $manual_username;
            $password = get_option('song_manager_db_password', '');
            $database = $manual_database;
            
            $dsn = "mysql:host={$host};port={$port};dbname={$database};charset=utf8mb4";
            $pdo = new PDO(
                $dsn,
                $username,
                $password,
                array(
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                )
            );

            // Ensure albums table exists when using manual DB
            $stmt2 = $pdo->query("SHOW TABLES LIKE 'albums'");
            if ($stmt2->rowCount() == 0) {
                song_manager_create_albums_table($pdo);
            }
            
            return $pdo;
        } catch(PDOException $e) {
            error_log("Song Manager manual database connection failed: " . $e->getMessage());
        }
    }
    $config_path = plugin_dir_path(__FILE__) . 'db.config.php';
    if (file_exists($config_path)) {
        require_once($config_path);
        
        if (defined('dbservername') && defined('dbusername') && defined('dbname')) {
            try {
                $host = dbservername;
                $port = defined('dbport') ? dbport : '3306';
                $username = dbusername;
                $password = defined('dbpassword') ? dbpassword : '';
                $database = dbname;
                
                $dsn = "mysql:host={$host};port={$port};dbname={$database};charset=utf8mb4";
                $pdo = new PDO(
                    $dsn,
                    $username,
                    $password,
                    array(
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_EMULATE_PREPARES => false
                    )
                );

                // Ensure albums table exists
                $stmt2 = $pdo->query("SHOW TABLES LIKE 'albums'");
                if ($stmt2->rowCount() == 0) {
                    song_manager_create_albums_table($pdo);
                }
                
                return $pdo;
            } catch(PDOException $e) {
                error_log("Song Manager config file database connection failed: " . $e->getMessage());
            }
        }
    }
    
    return false;
}

function song_manager_upload_song_file($file, $album_title = 'misc') {
    if (empty($file) || empty($file['name'])) {
        return new WP_Error('no_file', 'No file provided.');
    }

    if (!function_exists('wp_handle_upload')) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
    }

    $album_folder = sanitize_title($album_title);
    if (!$album_folder) {
        $album_folder = 'misc';
    }
    $subdir = '/music/' . $album_folder;
    $upload_dir_filter = function($dirs) use ($subdir) {
        $dirs['path'] = $dirs['basedir'] . $subdir;
        $dirs['url']  = $dirs['baseurl'] . $subdir;
        $dirs['subdir'] = $subdir;
        return $dirs;
    };
    add_filter('upload_dir', $upload_dir_filter);
    $overrides = array(
        'test_form' => false 
    );
    $movefile = wp_handle_upload($file, $overrides);
    remove_filter('upload_dir', $upload_dir_filter);
    if (isset($movefile['error'])) {
        return new WP_Error('upload_error', $movefile['error']);
    }

    if (empty($movefile['url'])) {
        return new WP_Error('upload_failed', 'Unknown upload error.');
    }

    return $movefile['url'];
}
add_action('wp_ajax_song_manager_async_upload', 'song_manager_async_upload');

function song_manager_async_upload() {
    if (!current_user_can('upload_files')) {
        wp_send_json_error('Permission denied.');
    }

    if (empty($_FILES['song_file']['name'])) {
        wp_send_json_error('No file uploaded.');
    }

    $album_name = isset($_POST['album']) ? sanitize_text_field($_POST['album']) : 'misc';
    $upload_result = song_manager_upload_song_file($_FILES['song_file'], $album_name);

    if (is_wp_error($upload_result)) {
        wp_send_json_error($upload_result->get_error_message());
    }

    wp_send_json_success(array('url' => $upload_result));
}



function song_manager_create_songs_table($pdo) {
    try {
        $sql = "CREATE TABLE songs (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            slug varchar(255) NOT NULL,
            artist varchar(255) DEFAULT '',
            genre varchar(100) DEFAULT '',
            album text DEFAULT NULL,
            cover text DEFAULT NULL,
            album_id mediumint(9) DEFAULT NULL,
            track mediumint(9) DEFAULT NULL,
            year year DEFAULT NULL,
            language varchar(50) DEFAULT '',
            audio_file varchar(500) DEFAULT '',
            lyrics text DEFAULT '',
            description text DEFAULT '',
            download_link varchar(500) DEFAULT '',
            buy_link varchar(500) DEFAULT '',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($sql);
        return true;
    } catch(PDOException $e) {
        error_log("Song Manager: Failed to create songs table: " . $e->getMessage());
        return false;
    }
}

function song_manager_create_albums_table($pdo) {
    try {
        $sql = "CREATE TABLE albums (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            artist varchar(255) DEFAULT '',
            cover varchar(500) DEFAULT '',
            genre varchar(100) DEFAULT '',
            year varchar(10) DEFAULT '',
            language varchar(50) DEFAULT '',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($sql);
        return true;
    } catch(PDOException $e) {
        error_log("Song Manager: Failed to create albums table: " . $e->getMessage());
        return false;
    }
}

add_action('admin_menu', 'song_manager_admin_menu');

function song_manager_admin_menu() {
    add_menu_page(
        'Song Manager',
        'Songs',
        'manage_options',
        'song-manager',
        'song_manager_admin_page',
        'dashicons-format-audio',
        6
    );
        add_submenu_page(
        'song-manager',
        'Add Song',
        'Add Song',
        'manage_options',
        'song-manager-add',
        'song_manager_add_page'
    );

    add_submenu_page(
        'song-manager',
        'Albums',
        'Albums',
        'manage_options',
        'song-manager-albums',
        'song_manager_albums_page'
    );
    
    add_submenu_page(
        'song-manager',
        'Tools',
        'Tools',
        'manage_options',
        'song-manager-tools',
        'song_manager_tools_page'
    );
    
    add_submenu_page(
        'song-manager',
        'Settings',
        'Settings',
        'manage_options',
        'song-manager-settings',
        'song_manager_settings_page'
    );

}

function song_manager_admin_page() {
    $pdo = song_manager_get_db_connection();
    global $pdo;
    if (!$pdo) {
        $pdo = song_manager_get_db_connection();
    }
    if (!$pdo) {
        echo '<div class="wrap">';
        echo '<h1>Song Manager</h1>';
        echo '<div class="error"><p>Database connection failed! Please check your <a href="' . admin_url('admin.php?page=song-manager-settings') . '">Settings</a>.</p></div>';
        echo '<p><a href="' . admin_url('admin.php?page=song-manager-settings') . '" class="button button-primary">Configure Database Settings</a></p>';
        echo '</div>';
        return;
    }
        if (isset($_GET['action']) && $_GET['action'] === 'deletesong' && isset($_GET['id'])) {
            $id = intval($_GET['id']);
            try {
                $stmt = $pdo->prepare("SELECT audio_file, cover FROM songs WHERE id = ?");
                $stmt->execute([$id]);
                $song = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($song) {
                    $upload_dir = wp_upload_dir();
                    $upload_base = trailingslashit($upload_dir['basedir']);
                    $site_url = trailingslashit(site_url());
                    if (!empty($song['audio_file'])) {
                        $file_path = str_replace($site_url, ABSPATH, $song['audio_file']);
                        if (strpos($file_path, $upload_base . 'music/') === 0 && file_exists($file_path)) {
                            @unlink($file_path);
                        }
                    }
                    if (!empty($song['cover']) && strpos($song['cover'], $upload_dir['baseurl']) === 0) {
                        $cover_path = str_replace($site_url, ABSPATH, $song['cover']);
                        if (strpos($cover_path, $upload_base . 'music/') === 0 && file_exists($cover_path)) {
                            @unlink($cover_path);
                        }
                    }
                }
                $stmt = $pdo->prepare("DELETE FROM songs WHERE id = ?");
                $stmt->execute([$id]);

                echo '<div class="updated"><p>Song deleted successfully!</p></div>';
            } catch (Exception $e) {
                echo '<div class="error"><p>Error deleting song: ' . esc_html($e->getMessage()) . '</p></div>';
            }
        }

        try {
            $stmt = $pdo->query("SELECT * FROM songs ORDER BY album_id, track ASC");
            $songs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            echo '<div class="error"><p>Error fetching songs: ' . esc_html($e->getMessage()) . '</p></div>';
            $songs = [];
        }
    ?>
    <div class="wrap">
        <h1>Song Manager</h1>
        <div style="margin-bottom: 20px; margin-top: 6px">
            <a href="<?php echo admin_url('admin.php?page=song-manager-add'); ?>" class="button button-primary">Add New Song</a>
        </div>
        <div style="margin-bottom: 20px; text-align: right;">
            <input type="text" id="song-filter" class="regular-text" placeholder="Type to filter…">
        </div>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Title</th>
                    <th>Artist</th>
                    <th>Genre</th>
                    <th>Album</th>
                    <th>Cover</th>
                    <th>Year</th>
                    <th>Language</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($songs)): ?>
                <tr>
                    <td colspan="8">No songs found. <a href="<?php echo admin_url('admin.php?page=song-manager-add'); ?>">Add your first song</a>!</td>
                </tr>
                <?php else: ?>
                <?php foreach ($songs as $song): ?>
                <tr>
                    <td><?php echo esc_html($song['id']); ?></td>
                    <td><?php echo esc_html($song['title']); ?></td>
                    <td><?php echo esc_html($song['artist']); ?></td>
                    <td><?php echo esc_html($song['genre']); ?></td>
                    <td><?php echo esc_html($song['album']); ?></td>
                    <td>
                        <?php if (!empty($song['cover'])): ?>
                            <img src="<?php echo esc_url($song['cover']); ?>" alt="<?php echo esc_attr($song['title']); ?> cover" style="width:50px;height:auto;display:block;">
                        <?php else: ?>
                            <span style="color:#aaa;">No cover</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo esc_html($song['year']); ?></td>
                    <td><?php echo esc_html($song['language']); ?></td>
                    <td>
                        <a href="<?php echo admin_url('admin.php?page=song-manager-add&edit=' . $song['id']); ?>">Edit</a> |
                        <a href="?page=song-manager&action=deletesong&id=<?php echo intval($song['id']); ?>" 
                            onclick="return confirm('Are you sure you want to delete this song?');">
                            Delete
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        const filterInput = document.getElementById('song-filter');
        const table = document.querySelector('.wp-list-table');
        if (!filterInput || !table) return;

        filterInput.addEventListener('keyup', function() {
            const query = this.value.toLowerCase().trim();
            const rows = table.querySelectorAll('tbody tr');

            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(query) ? '' : 'none';
            });
        });
    });
    </script>
                     
     <?php
}

function song_manager_add_page() {
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        echo '<div class="error"><p>Database connection failed!</p></div>';
        return;
    }
    
    $editing = false;
    $song = null;
    
    if (isset($_GET['edit'])) {
        $editing = true;
        $id = intval($_GET['edit']);
        $stmt = $pdo->prepare("SELECT * FROM songs WHERE id = ?");
        $stmt->execute([$id]);
        $song = $stmt->fetch(PDO::FETCH_ASSOC);
    }

    if ($_POST) {
        $title = sanitize_text_field($_POST['title']);
        $slug = sanitize_title($_POST['slug'] ?: $title);
        $artist = sanitize_text_field($_POST['artist']);
        $genre = sanitize_text_field($_POST['genre']);
        $album_input = sanitize_text_field($_POST['album'] ?? '');
        $album_id_input = isset($_POST['album_id']) ? intval($_POST['album_id']) : null;
        $track = intval($_POST['track']) ?: null;
        $year = intval($_POST['year']) ?: null;
        $language = sanitize_text_field($_POST['language']);
$audio_file = '';
if (!empty($_POST['audio_file'])) {
    $audio_file = esc_url_raw($_POST['audio_file']);
}
$download_link = $audio_file;   

if (!empty($_FILES['song_file']['name'])) {
    // Determine album title to build folder: use album_title_for_db if available (you compute it earlier)
    // Fallback to posted album text or 'misc'
    $album_for_folder = '';
    if (!empty($album_title_for_db)) {
        $album_for_folder = $album_title_for_db;
    } elseif (!empty($_POST['album'])) {
        $album_for_folder = sanitize_text_field($_POST['album']);
    } elseif (!empty($album_id_input)) {
        // try to get album from DB if album_id provided (best-effort)
        try {
            $pdo_tmp = song_manager_get_db_connection();
            if ($pdo_tmp) {
                $stmt_tmp = $pdo_tmp->prepare('SELECT title FROM albums WHERE id = ?');
                $stmt_tmp->execute([$album_id_input]);
                $tmpAlb = $stmt_tmp->fetch(PDO::FETCH_ASSOC);
                if ($tmpAlb && !empty($tmpAlb['title'])) {
                    $album_for_folder = $tmpAlb['title'];
                }
            }
        } catch (Exception $e) {
            // ignore and fallback
        }
    }

    if (!$album_for_folder) {
        $album_for_folder = 'misc';
    }

    $upload_result = song_manager_upload_song_file($_FILES['song_file'], $album_for_folder);

    if (is_wp_error($upload_result)) {
        echo '<div class="notice notice-error"><p>Upload failed: ' . esc_html($upload_result->get_error_message()) . '</p></div>';
    } else {
        $audio_file = esc_url_raw($upload_result);
        $download_link = $audio_file; // store same URL
    }
}

        $cover = esc_url_raw($_POST['cover']);
        $lyrics = wp_kses_post($_POST['lyrics']);
        $description = wp_kses_post($_POST['description']);
        $buy_link = esc_url_raw($_POST['buy_link']);

        // If album_id provided, try to fetch album title and set album field to the album title for compatibility
        $album_title_for_db = $album_input;
        if ($album_id_input) {
            try {
                $stmtAlbum = $pdo->prepare("SELECT title FROM albums WHERE id = ?");
                $stmtAlbum->execute([$album_id_input]);
                $alb = $stmtAlbum->fetch(PDO::FETCH_ASSOC);
                if ($alb && !empty($alb['title'])) {
                    $album_title_for_db = $alb['title'];
                }
            } catch (Exception $e) {
                // ignore, fallback to album_input
            }
        }

        try {
            if ($editing) {
                $stmt = $pdo->prepare("UPDATE songs SET title=?, slug=?, artist=?, genre=?, album=?, album_id=?, track=?, year=?, language=?, audio_file=?, cover=?, lyrics=?, description=?, download_link=?, buy_link=? WHERE id=?");
                $stmt->execute([$title, $slug, $artist, $genre, $album_title_for_db, $album_id_input, $track, $year, $language, $audio_file, $cover, $lyrics, $description, $download_link, $buy_link, $song['id']]);
                echo '<div class="updated"><p>Song updated successfully!</p></div>';
            } else {
$stmt = $pdo->prepare("
    INSERT INTO songs (
        title, slug, artist, genre, album, album_id, track, year, language,
        audio_file, cover, lyrics, description, download_link, buy_link
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
");

$stmt->execute([
    $title,
    $slug,
    $artist,
    $genre,
    $album_title_for_db,
    $album_id_input,
    $track,
    $year,
    $language,
    $audio_file,
    $cover,
    $lyrics,
    $description,
    $download_link,
    $buy_link
]);

echo '<div class="updated"><p>Song added successfully!</p></div>';

            }

            if ($editing) {
                $stmt = $pdo->prepare("SELECT * FROM songs WHERE id = ?");
                $stmt->execute([$song['id']]);
                $song = $stmt->fetch(PDO::FETCH_ASSOC);
            }
        } catch (Exception $e) {
            echo '<div class="error"><p>Error: ' . esc_html($e->getMessage()) . '</p></div>';
        }
    }

    // Fetch albums for the dropdown
    try {
        $albums = $pdo->query("SELECT id, title FROM albums ORDER BY title")->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $albums = [];
    }
    
    ?>
    <div class="wrap">
        <h1><?php echo $editing ? 'Edit Song' : 'Add New Song'; ?></h1>
        <form method="post" action="" enctype="multipart/form-data">
            <table class="form-table">
                <tr>
                    <th scope="row">Track</th>
                    <td><input type="text" name="track" value="<?php echo $song ? esc_attr($song['track']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Title *</th>
                    <td><input type="text" name="title" value="<?php echo $song ? esc_attr($song['title']) : ''; ?>" class="regular-text" required /></td>
                </tr>
                <tr>
                    <th scope="row">Slug</th>
                    <td>
                        <input type="text" name="slug" value="<?php echo $song ? esc_attr($song['slug']) : ''; ?>" class="regular-text" />
                        <p class="description">Leave empty to auto-generate from title</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Artist</th>
                    <td><input type="text" name="artist" value="<?php echo $song ? esc_attr($song['artist']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Genre</th>
                    <td><input type="text" name="genre" value="<?php echo $song ? esc_attr($song['genre']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <!-- Album: searchable list + hidden album_id (Option A - keep text field too) -->
                <tr>
                    <th scope="row">Album</th>
                    <td>
                        <div class="form-group">
                            <label for="album-search">Search Albums:</label>
                            <input id="album-search" class="regular-text" placeholder="Filter…">

                            <fieldset style="border:1px solid #ddd;padding:8px;margin-top:8px;max-height:220px;overflow:auto;">
                              <legend style="font-size:12px;padding:0 4px;">Albums (click to select)</legend>
                              <div class="checkbox-container" id="album-container" style="display:block;">
                                <?php
                                    // fetch albums (id, title, artist, genre, year, language, cover)
                                    try {
                                        $albums_list = $pdo->query("SELECT id, title, artist, genre, year, language, cover FROM albums ORDER BY title")->fetchAll(PDO::FETCH_ASSOC);
                                    } catch (Exception $e) {
                                        $albums_list = [];
                                    }
                                    if (!empty($albums_list)):
                                        foreach ($albums_list as $alb):
                                            // ensure safe attributes
                                            $a_id = intval($alb['id']);
                                            $a_title = esc_attr($alb['title']);
                                            $a_artist = esc_attr($alb['artist']);
                                            $a_genre = esc_attr($alb['genre']);
                                            $a_year = esc_attr($alb['year']);
                                            $a_language = esc_attr($alb['language']);
                                            $a_cover = esc_url($alb['cover']);
                                            ?>
                                            <label class="album-item" style="display:block;padding:6px;border-bottom:1px dashed #eee;cursor:pointer;" data-id="<?php echo $a_id; ?>" data-title="<?php echo $a_title; ?>" data-artist="<?php echo $a_artist; ?>" data-genre="<?php echo $a_genre; ?>" data-year="<?php echo $a_year; ?>" data-language="<?php echo $a_language; ?>" data-cover="<?php echo $a_cover; ?>">
                                                <input type="radio" name="album_radio" value="<?php echo $a_id; ?>" style="margin-right:6px;">
                                                <strong style="display:inline-block;min-width:200px;"><?php echo esc_html($alb['title']); ?></strong>
                                                <span style="color:#777;margin-left:6px;"><?php echo esc_html($alb['artist']); ?></span>
                                            </label>
                                        <?php
                                        endforeach;
                                    else:
                                        ?>
                                        <div style="color:#888;padding:8px;">No albums available. Add one on the Albums page.</div>
                                    <?php endif; ?>
                              </div>
                            </fieldset>
                        </div>
                        <p class="description">Click an album to fill the song fields. You can still edit any field afterwards.</p>

                        <!-- hidden album_id to store link -->
                        <input type="hidden" name="album_id" id="album_id" value="<?php echo $song ? esc_attr($song['album_id']) : ''; ?>" />
                    </td>
                </tr>
                <!-- Text album field (kept for compatibility and manual entries) -->
                <tr>
                    <th scope="row">Album (text)</th>
                    <td><input type="text" name="album" value="<?php echo $song ? esc_attr($song['album']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <!-- Cover (URL + Media Library button) -->
                <tr>
                    <th scope="row">Cover</th>
                    <td>
                        <input type="url" name="cover" id="cover" value="<?php echo $song ? esc_attr($song['cover']) : ''; ?>" class="regular-text" />
                        <button type="button" class="button" id="upload_cover_button">Upload / Select Image</button>
                        <p class="description">Use the button to pick an image from the Media Library (the URL will be stored).</p>
                        <div id="cover_preview" style="margin-top:10px;">
                            <?php if ($song && !empty($song['cover'])): ?>
                                <img src="<?php echo esc_url($song['cover']); ?>" style="max-width:100px; height:auto;">
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>

                <tr>
                    <th scope="row">Year</th>
                    <td><input type="number" name="year" value="<?php echo $song ? esc_attr($song['year']) : ''; ?>" min="1900" max="2099" /></td>
                </tr>
                <tr>
                    <th scope="row">Language</th>
                    <td><input type="text" name="language" value="<?php echo $song ? esc_attr($song['language']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                <th scope="row">Song file</th>
                <td>
                    <div id="upload-container">
                    <!-- disabled until album is provided -->
                    <input type="file" id="song-file" accept="audio/*" disabled />
                    <span id="upload-disabled-hint" class="description" style="margin-left:8px;">Upload enabled after album is selected or entered.</span>

                    <div id="upload-progress" style="display:none;width:100%;background:#eee;border-radius:4px;margin-top:8px;">
                        <div id="upload-bar" style="height:18px;width:0%;background:#0073aa;border-radius:4px;"></div>
                    </div>

                    <input type="hidden" name="audio_file" id="audio_file" value="<?php echo esc_attr($song ? $song['audio_file'] : ''); ?>">

                    <p id="upload-status" class="description">
                        <?php if ($song && !empty($song['audio_file'])): ?>
                        Current: <a href="<?php echo esc_url($song['audio_file']); ?>" target="_blank"><?php echo basename($song['audio_file']); ?></a>
                        <?php else: ?>
                        Select or enter an album to enable upload.
                        <?php endif; ?>
                    </p>
                    </div>
                </td>
                </tr>

                <tr>
                    <th scope="row">Lyrics</th>
                    <td><textarea name="lyrics" rows="10" cols="50"><?php echo $song ? esc_textarea($song['lyrics']) : ''; ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row">Description</th>
                    <td><textarea name="description" rows="5" cols="50"><?php echo $song ? esc_textarea($song['description']) : ''; ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row">Buy Link</th>
                    <td><input type="url" name="buy_link" value="<?php echo $song ? esc_attr($song['buy_link']) : ''; ?>" class="regular-text" /></td>
                </tr>
            </table>
            
            <?php submit_button($editing ? 'Update Song' : 'Add Song'); ?>
        </form>
    </div>
    <?php
}

function song_manager_settings_page() {
    if (isset($_POST['save_db_settings'])) {
        update_option('song_manager_db_host', sanitize_text_field($_POST['db_host']));
        update_option('song_manager_db_port', sanitize_text_field($_POST['db_port']));
        update_option('song_manager_db_name', sanitize_text_field($_POST['db_name']));
        update_option('song_manager_db_username', sanitize_text_field($_POST['db_username']));
        update_option('song_manager_db_password', $_POST['db_password']);
        update_option('song_manager_use_wp_db', isset($_POST['use_wp_db']) ? 1 : 0);
        update_option('song_manager_songs_per_page', intval($_POST['songs_per_page'])); // ✅ new line
        echo '<div class="updated"><p>Settings saved successfully!</p></div>';
    }

    
    $db_host = get_option('song_manager_db_host', 'localhost');
    $db_port = get_option('song_manager_db_port', '3306');
    $db_name = get_option('song_manager_db_name', '');
    $db_username = get_option('song_manager_db_username', '');
    $db_password = get_option('song_manager_db_password', '');
    $use_wp_db = get_option('song_manager_use_wp_db', 0);
    $connection_status = song_manager_test_db_connection();
    
    ?>
    <div class="wrap">
        <h1>Song Manager Settings</h1>
        <div class="notice <?php echo $connection_status['success'] ? 'notice-success' : 'notice-error'; ?>">
            <p><strong>Connection Status:</strong> <?php echo $connection_status['message']; ?></p>
        </div>
        <form method="post" action="">
            <h3>Display Settings</h3>
            <table class="form-table">
                <tr>
                    <th scope="row">Songs per Page</th>
                    <td>
                        <?php 
                            $songs_per_page = get_option('song_manager_songs_per_page', 10); 
                        ?>
                        <input type="number" name="songs_per_page" value="<?php echo esc_attr($songs_per_page); ?>" min="1" class="small-text" />
                        <p class="description">Number of songs to display per page in the frontend “normal” view.</p>
                    </td>
                </tr>
            </table>       
            <div class="card">
                <table class="form-table">
                    <tr>
                        <th scope="row">Use WordPress Database</th>
                        <td>
                            <input type="checkbox" name="use_wp_db" value="1" <?php checked($use_wp_db, 1); ?> id="use_wp_db" />
                            <label for="use_wp_db">Use WordPress database (<?php echo DB_NAME; ?>) instead of separate database</label>
                            <p class="description">Recommended: This creates a 'songs' table in your WordPress database.</p>
                        </td>
                    </tr>
                </table>
                <h3>Separate Database Settings</h3>
                <p class="description">Only used if "Use WordPress Database" is not checked above.</p>
            
                <table class="form-table">
                    <tr>
                        <th scope="row">Database Host</th>
                        <td><input type="text" name="db_host" value="<?php echo esc_attr($db_host); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Database Port</th>
                        <td><input type="text" name="db_port" value="<?php echo esc_attr($db_port); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Database Name</th>
                        <td><input type="text" name="db_name" value="<?php echo esc_attr($db_name); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Database Username</th>
                        <td><input type="text" name="db_username" value="<?php echo esc_attr($db_username); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Database Password</th>
                        <td><input type="password" name="db_password" value="<?php echo esc_attr($db_password); ?>" class="regular-text" /></td>
                    </tr>
                </table>
            </div>
        <?php submit_button('Save Settings', 'primary', 'save_db_settings'); ?>
        </form>
    </div>
    
    <script>
    document.getElementById('use_wp_db').addEventListener('change', function() {
        var separateDbFields = document.querySelectorAll('input[name="db_host"], input[name="db_port"], input[name="db_name"], input[name="db_username"], input[name="db_password"]');
        separateDbFields.forEach(function(field) {
            field.disabled = this.checked;
            field.style.opacity = this.checked ? '0.5' : '1';
        }.bind(this));
    });

    document.getElementById('use_wp_db').dispatchEvent(new Event('change'));
    </script>
    <?php
}

function song_manager_test_db_connection() {
    $pdo = song_manager_get_db_connection();
    
    if (!$pdo) {
        return [
            'success' => false,
            'message' => '❌ Database connection failed. Please check your settings.'
        ];
    }
    
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE 'songs'");
        $table_exists = $stmt->rowCount() > 0;
        
        if (!$table_exists) {
            return [
                'success' => false,
                'message' => '⚠️ Connected but songs table does not exist. Please deactivate and reactivate the plugin.'
            ];
        }
        
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM songs");
        $count = $stmt->fetch()['count'];
        
        return [
            'success' => true,
            'message' => "✅ Connected successfully. Songs table exists with {$count} songs."
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '❌ Connection error: ' . $e->getMessage()
        ];
    }
}

function song_manager_tools_page() {
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        echo '<div class="error"><p>Database connection failed!</p></div>';
        return;
    }
    if (isset($_POST['export_songs'])) {
        $stmt = $pdo->query("SELECT * FROM songs ORDER BY created_at DESC");
        $songs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="songs_export_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        if (!empty($songs)) {
            fputcsv($output, array_keys($songs[0]));
            foreach ($songs as $song) {
                fputcsv($output, $song);
            }
        }
        fclose($output);
        exit;
    }
    
    ?>
    <div class="wrap">
        <h1>Song Manager Tools</h1>
        
        <div class="card">
            <h2>Export Songs</h2>
            <p>Export all songs to a CSV file for backup or migration purposes.</p>
            <form method="post">
                <input type="submit" name="export_songs" class="button button-primary" value="Export Songs to CSV" />
            </form>
        </div>
        
        <div class="card">
            <h2>Database Info</h2>
            <?php
            try {
                $stmt = $pdo->query("SELECT COUNT(*) as count FROM songs");
                $count = $stmt->fetch()['count'];
                echo '<p><strong>Total songs:</strong> ' . $count . '</p>';
                
                $stmt = $pdo->query("SELECT COUNT(DISTINCT genre) as count FROM songs WHERE genre != ''");
                $genre_count = $stmt->fetch()['count'];
                echo '<p><strong>Unique genres:</strong> ' . $genre_count . '</p>';
                
                echo '<p><strong>Database status:</strong> ✅ Connected</p>';
            } catch (Exception $e) {
                echo '<p><strong>Database status:</strong> ❌ Error: ' . esc_html($e->getMessage()) . '</p>';
            }
            ?>
        </div>
    </div>
    <?php
}

function song_manager_albums_page() {
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        echo '<div class="error"><p>Database connection failed!</p></div>';
        return;
    }

    $editing = false;
    $album = null;

    // Handle delete
    if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
        $id = intval($_GET['id']);
        try {
            $stmt = $pdo->prepare("DELETE FROM albums WHERE id = ?");
            $stmt->execute([$id]);
            echo '<div class="updated"><p>Album deleted successfully!</p></div>';
        } catch (Exception $e) {
            echo '<div class="error"><p>Error deleting album: ' . esc_html($e->getMessage()) . '</p></div>';
        }
    }

    // Edit mode
    if (isset($_GET['edit'])) {
        $editing = true;
        $id = intval($_GET['edit']);
        $stmt = $pdo->prepare("SELECT * FROM albums WHERE id = ?");
        $stmt->execute([$id]);
        $album = $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Handle form submission for add/edit
    if ($_POST) {
        $title = sanitize_text_field($_POST['title']);
        $artist = sanitize_text_field($_POST['artist']);
        $cover = esc_url_raw($_POST['cover']);
        $genre = sanitize_text_field($_POST['genre']);
        $year = sanitize_text_field($_POST['year']);
        $language = sanitize_text_field($_POST['language']);

        try {
            if ($editing && $album) {
                $stmt = $pdo->prepare("UPDATE albums SET title=?, artist=?, cover=?, genre=?, year=?, language=? WHERE id=?");
                $stmt->execute([$title, $artist, $cover, $genre, $year, $language, $album['id']]);
                echo '<div class="updated"><p>Album updated successfully!</p></div>';
            } else {
                $stmt = $pdo->prepare("INSERT INTO albums (title, artist, cover, genre, year, language) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$title, $artist, $cover, $genre, $year, $language]);
                echo '<div class="updated"><p>Album added successfully!</p></div>';
            }

            // Refresh album data if editing
            if ($editing) {
                $stmt = $pdo->prepare("SELECT * FROM albums WHERE id = ?");
                $stmt->execute([$album['id']]);
                $album = $stmt->fetch(PDO::FETCH_ASSOC);
            }
        } catch (Exception $e) {
            echo '<div class="error"><p>Error saving album: ' . esc_html($e->getMessage()) . '</p></div>';
        }
    }

    // Fetch albums for list
    try {
        $stmt = $pdo->query("SELECT * FROM albums ORDER BY created_at DESC");
        $albums = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $albums = [];
        echo '<div class="error"><p>Error loading albums: ' . esc_html($e->getMessage()) . '</p></div>';
    }

    ?>
    <div class="wrap">
        <h1>Albums</h1>

        <div style="margin-bottom:20px;">
            <a href="<?php echo admin_url('admin.php?page=song-manager-albums'); ?>" class="button">Albums List</a>
        </div>

        <h2><?php echo $editing ? 'Edit Album' : 'Add New Album'; ?></h2>

        <form method="post" action="">
            <table class="form-table">
                <tr>
                    <th scope="row">Title *</th>
                    <td><input type="text" name="title" value="<?php echo $album ? esc_attr($album['title']) : ''; ?>" class="regular-text" required /></td>
                </tr>
                <tr>
                    <th scope="row">Artist</th>
                    <td><input type="text" name="artist" value="<?php echo $album ? esc_attr($album['artist']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Genre</th>
                    <td><input type="text" name="genre" value="<?php echo $album ? esc_attr($album['genre']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Year</th>
                    <td><input type="text" name="year" value="<?php echo $album ? esc_attr($album['year']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Language</th>
                    <td><input type="text" name="language" value="<?php echo $album ? esc_attr($album['language']) : ''; ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Cover</th>
                    <td>
                        <input type="url" name="cover" id="album_cover" value="<?php echo $album ? esc_attr($album['cover']) : ''; ?>" class="regular-text" />
                        <button type="button" class="button" id="upload_album_cover_button">Upload / Select Image</button>
                        <p class="description">Use the button to pick an image from the Media Library (the URL will be stored).</p>
                        <div id="album_cover_preview" style="margin-top:10px;">
                            <?php if ($album && !empty($album['cover'])): ?>
                                <img src="<?php echo esc_url($album['cover']); ?>" style="max-width:100px; height:auto;">
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            </table>

            <?php submit_button($editing ? 'Update Album' : 'Add Album'); ?>
        </form>

        <hr>

        <h2>Existing Albums</h2>
        <table class="widefat striped">
            <thead>
                <tr>
                    <th>Cover</th>
                    <th>Title</th>
                    <th>Artist</th>
                    <th>Genre</th>
                    <th>Year</th>
                    <th>Language</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($albums)): ?>
                    <tr><td colspan="7">No albums found.</td></tr>
                <?php else: ?>
                    <?php foreach ($albums as $al): ?>
                        <tr>
                            <td>
                                <?php if (!empty($al['cover'])): ?>
                                    <img src="<?php echo esc_url($al['cover']); ?>" style="width:50px;height:auto;display:block;">
                                <?php else: ?>
                                    <span style="color:#aaa;">No cover</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html($al['title']); ?></td>
                            <td><?php echo esc_html($al['artist']); ?></td>
                            <td><?php echo esc_html($al['genre']); ?></td>
                            <td><?php echo esc_html($al['year']); ?></td>
                            <td><?php echo esc_html($al['language']); ?></td>
                            <td>
                                <a href="<?php echo admin_url('admin.php?page=song-manager-albums&edit=' . $al['id']); ?>">Edit</a> |
                                <a href="<?php echo admin_url('admin.php?page=song-manager-albums&action=delete&id=' . $al['id']); ?>" onclick="return confirm('Are you sure?')">Delete</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <?php
}

add_shortcode('songs', 'song_manager_display_songs');
add_shortcode('music', 'song_manager_display_songs');

function song_manager_display_songs($atts) {
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        return '<p>Database connection failed!</p>';
    }
     $atts = shortcode_atts([
        'filter' => '1',
        'view' => '1',
        'limit' => '0'
    ], $atts, 'song_manager');

    $show_filters = $atts['filter'] == '1';   
    $view_mode = isset($atts['view']) ? $atts['view'] : 'normal';
    $limit = isset($atts['limit']) ? $atts['limit'] : '0';
    ob_start();
    ?>
    <div class="song-manager-container">
        <?php if ($show_filters): ?>
        <!-- Search and Filter Section -->
            <div class="song-filters">
                <div class="search-box">
                    <input type="text" id="song-search" placeholder="🔍 Search songs..." />
                </div>

                <div class="filter-buttons">
                    <select id="album-filter">
                        <option value="">💿 All Albums</option>
                    </select>  
                    <select id="genre-filter">
                        <option value="">🎵 All Genres</option>
                    </select>
                    <select id="year-filter">
                        <option value="">📅 All Years</option>
                    </select>
                    <select id="language-filter">
                        <option value="">🗣️ All Languages</option>
                    </select>
                </div>
            </div>
         <?php endif; ?>
        <!-- Songs Grid -->
        <div class="songs-grid" id="songs-grid">
            <!-- Songs will be loaded here via JavaScript -->
        </div>
        <div id="pagination-controls" class="pagination-controls"></div>
    </div>
    <div id="song-modal" class="song-modal">
        <div class="song-modal-content">
            <span class="song-modal-close">&times;</span>
            <div id="song-modal-body">
                <!-- Song details will be loaded here -->
            </div>
        </div>
    </div>   
    <script>
        const SONG_VIEW_MODE = <?php echo json_encode($view_mode); ?>;
        const SONG_LIMIT = <?php echo json_encode($limit); ?>; // integer, safe
        document.addEventListener('DOMContentLoaded', function() {
            loadSongs();
            setupFilters();
            setupModal();
        });
    
    let currentPage = 1;
    let totalPages = 1;

    function loadSongs(filters = {}) {
        const formData = new FormData();
        formData.append('action', 'get_songs');
        formData.append('nonce', '<?php echo wp_create_nonce("song_manager_nonce"); ?>');
        formData.append('page', currentPage);
        formData.append('limit', SONG_LIMIT);

        if (filters.search) formData.append('search', filters.search);
        if (filters.genre) formData.append('genre', filters.genre);
        if (filters.year) formData.append('year', filters.year);
        if (filters.language) formData.append('language', filters.language);
        if (filters.album) formData.append('album', filters.album);

        fetch('<?php echo admin_url("admin-ajax.php"); ?>', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displaySongs(data.data.songs);
                updateFilterOptions(data.data.filters);
                totalPages = data.data.pagination?.total_pages || 1;
                currentPage = data.data.pagination?.current_page || 1;
                renderPagination();
            }
        })
        .catch(error => console.error('Error:', error));
    }

    function renderPagination() {
        const pagination = document.getElementById('pagination-controls');
        if (totalPages <= 1) {
            pagination.innerHTML = '';
            return;
        }

        let buttons = '';
        for (let i = 1; i <= totalPages; i++) {
            buttons += `<button class="pagination-btn ${i === currentPage ? 'active' : ''}" onclick="gotoPage(${i})">${i}</button>`;
        }
        pagination.innerHTML = buttons;
    }

    function gotoPage(page) {
        currentPage = page;
        loadSongs();
    }


        function displaySongs(songs) {
                const grid = document.getElementById('songs-grid');
                if (songs.length === 0) {
                    grid.innerHTML = '<p>No songs found matching your criteria.</p>';
                    return;
                }

            if (SONG_VIEW_MODE === 'small') {
                // Small compact view: title + player inline
                grid.innerHTML = songs.map(song => `
                    <div class="song-row">
                        <span class="song-title-small">${song.title}</span>
                        ${song.audio_file ? `
                            <audio controls class="song-audio-small" onclick="event.stopPropagation()">
                                <source src="${song.audio_file}" type="audio/mpeg">
                                Your browser does not support the audio element.
                            </audio>
                        ` : ''}
                    </div>
                `).join('');
            } else {
                grid.innerHTML = songs.map(song => `
                    <div class="song-card" onclick="openSongModal(${song.id})">
                        ${song.cover ? `<img src="${song.cover}" alt="${song.title} Cover" class="song-cover">` : ''}
                        <div class="song-title">${song.title}</div>
                        <div class="song-artist">${song.artist}</div>
                        <div class="song-meta">
                            <span>🎵 ${song.genre}</span>
                            <span>📅 ${song.year}</span>
                            <span>🗣️ ${song.language}</span>
                        </div>
                        ${song.audio_file ? `
                            <div class="song-audio">
                                <audio controls onclick="event.stopPropagation()">
                                    <source src="${song.audio_file}" type="audio/mpeg">
                                    Your browser does not support the audio element.
                                </audio>
                            </div>
                        ` : ''}
                        <div class="song-actions" onclick="event.stopPropagation()">
                            <button class="btn btn-primary" onclick="openSongModal(${song.id})">View Details</button>
                            ${song.download_link ? `<a href="${song.download_link}" class="btn btn-secondary">⬇️ Download</a>` : ''}
                        </div>
                    </div>
                `).join('');
            }
        }  
        function updateFilterOptions(filters) {
            const genreSelect = document.getElementById('genre-filter');
            const yearSelect = document.getElementById('year-filter');
            const languageSelect = document.getElementById('language-filter');
            const albumSelect = document.getElementById('album-filter'); 
            genreSelect.innerHTML = '<option value="">🎵 All Genres</option>';
            filters.genres.forEach(genre => {
                if (genre) genreSelect.innerHTML += `<option value="${genre}">${genre}</option>`;
            });
            yearSelect.innerHTML = '<option value="">📅 All Years</option>';
            filters.years.forEach(year => {
                if (year) yearSelect.innerHTML += `<option value="${year}">${year}</option>`;
            });
            languageSelect.innerHTML = '<option value="">🗣️ All Languages</option>';
            filters.languages.forEach(language => {
                if (language) languageSelect.innerHTML += `<option value="${language}">${language}</option>`;
            });
            albumSelect.innerHTML = '<option value="">💿 All Albums</option>';
            filters.albums.forEach(album => {
                if (album) albumSelect.innerHTML += `<option value="${album}">${album}</option>`;
            });
        }
            
        function setupFilters() {
            const searchInput = document.getElementById('song-search');
            const genreSelect = document.getElementById('genre-filter');
            const yearSelect = document.getElementById('year-filter');
            const languageSelect = document.getElementById('language-filter');
            const albumSelect = document.getElementById('album-filter'); // added album filter

            let timeout;

            function handleFilter() {
                clearTimeout(timeout);
                timeout = setTimeout(() => {
                    const filters = {
                        search: searchInput.value,
                        genre: genreSelect.value,
                        year: yearSelect.value,
                        language: languageSelect.value,
                        album: albumSelect.value
                    };
                    loadSongs(filters);
                }, 300);
            }
            searchInput.addEventListener('input', handleFilter);
            genreSelect.addEventListener('change', handleFilter);
            yearSelect.addEventListener('change', handleFilter);
            languageSelect.addEventListener('change', handleFilter);
            albumSelect.addEventListener('change', handleFilter);
        }
        function setupModal() {
            const modal = document.getElementById('song-modal');
            const closeBtn = document.querySelector('.song-modal-close');
            closeBtn.onclick = function() {
                modal.style.display = 'none';
                const audio = modal.querySelector('audio');
                if (audio) {
                    audio.pause();
                }
            }
            window.onclick = function(event) {
                if (event.target === modal) {
                    modal.style.display = 'none';
                    const audio = modal.querySelector('audio');
                    if (audio) {
                        audio.pause();
                    }
                }
            }
            document.addEventListener('keydown', function(event) {
                if (event.key === 'Escape' && modal.style.display === 'block') {
                    modal.style.display = 'none';
                    const audio = modal.querySelector('audio');
                    if (audio) {
                        audio.pause();
                    }
                }
            });
        }
        
        function openSongModal(songId) {
            const formData = new FormData();
            formData.append('action', 'get_song_details');
            formData.append('song_id', songId);
            formData.append('nonce', '<?php echo wp_create_nonce("song_manager_nonce"); ?>');
            
            fetch('<?php echo admin_url("admin-ajax.php"); ?>', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    displaySongModal(data.data);
                }
            })
            .catch(error => console.error('Error:', error));
        }
        
        function displaySongModal(song) {
            const modalBody = document.getElementById('song-modal-body');
            
            modalBody.innerHTML = `
                <div class="modal-song-header">
                    <h1 class="modal-song-title">${song.title}</h1>
                    <p class="modal-song-artist">by ${song.artist}</p>
                    <div class="modal-song-meta">
                        <span>🎵 ${song.genre}</span>
                        <span>📅 ${song.year}</span>
                        <span>🗣️ ${song.language}</span>
                    </div>
                </div>
                
                ${song.cover ? `
                    <div class="modal-song-cover">
                        <img src="${song.cover}" alt="${song.title} cover">
                    </div>
                ` : ''}
                
                ${song.audio_file ? `
                    <div class="modal-song-player">
                        <h3>🎧 Player</h3>
                        <audio controls>
                            <source src="${song.audio_file}" type="audio/mpeg">
                            Your browser does not support the audio element.
                        </audio>
                    </div>
                ` : ''}
                
                ${song.lyrics ? `
                    <div class="modal-song-lyrics">
                        <h3>📜 Lyrics</h3>
                        <div class="modal-lyrics-content">${song.lyrics.replace(/\n/g, '<br>')}</div>
                    </div>
                ` : ''}
                
                ${song.description ? `
                    <div class="modal-song-description">
                        <h3>ℹ️ Description</h3>
                        <div class="modal-description-content">${song.description.replace(/\n/g, '<br>')}</div>
                    </div>
                ` : ''}
                
                <div class="modal-song-actions">
                    ${song.download_link ? `<a href="${song.download_link}" class="modal-btn modal-btn-download">⬇️ Download</a>` : ''}
                    ${song.buy_link ? `<a href="${song.buy_link}" class="modal-btn modal-btn-buy">💰 Buy</a>` : ''}
                </div>
            `;
            
            document.getElementById('song-modal').style.display = 'block';
            document.querySelector('.song-modal-content').scrollTop = 0;
        }
    </script>
    <?php
    
    return ob_get_clean();
}

add_action('wp_ajax_get_songs', 'song_manager_ajax_get_songs');
add_action('wp_ajax_nopriv_get_songs', 'song_manager_ajax_get_songs');

function song_manager_ajax_get_songs() {
    if (!wp_verify_nonce($_POST['nonce'], 'song_manager_nonce')) {
        wp_die('Security check failed');
    }

    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        wp_send_json_error('Database connection failed');
    }

    $search   = isset($_POST['search'])   ? sanitize_text_field($_POST['search'])   : '';
    $genre    = isset($_POST['genre'])    ? sanitize_text_field($_POST['genre'])    : '';
    $year     = isset($_POST['year'])     ? intval($_POST['year'])                  : '';
    $language = isset($_POST['language']) ? sanitize_text_field($_POST['language']) : '';
    $album    = isset($_POST['album'])    ? sanitize_text_field($_POST['album'])    : '';

    $where_conditions = [];
    $params = [];

if ($search) {
    $where_conditions[] = "(title LIKE :search_title OR artist LIKE :search_artist OR lyrics LIKE :search_lyrics)";
    $params[':search_title']  = '%' . $search . '%';
    $params[':search_artist'] = '%' . $search . '%';
    $params[':search_lyrics'] = '%' . $search . '%';
}


    if ($genre) {
        $where_conditions[] = "genre = :genre";
        $params[':genre'] = $genre;
    }

    if ($year) {
        $where_conditions[] = "year = :year";
        $params[':year'] = $year;
    }

    if ($language) {
        $where_conditions[] = "language = :language";
        $params[':language'] = $language;
    }

    if ($album) {
        $where_conditions[] = "album = :album";
        $params[':album'] = $album;
    }

    // Pagination
    $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
    $songs_per_page = intval(get_option('song_manager_songs_per_page', 10));
    if ($songs_per_page <= 0) $songs_per_page = 10;

    $offset = ($page - 1) * $songs_per_page;
    $where_clause = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

    // Count total
    $count_sql = "SELECT COUNT(*) FROM songs $where_clause";
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_songs = (int) $count_stmt->fetchColumn();

    // Fetch songs
    $sql = "SELECT * FROM songs $where_clause ORDER BY album_id, track ASC LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);

    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $songs_per_page, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);

    $stmt->execute();
    $songs = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Filters
    $genres = $pdo->query("SELECT DISTINCT genre FROM songs WHERE genre != '' ORDER BY genre")->fetchAll(PDO::FETCH_COLUMN);
    $years = $pdo->query("SELECT DISTINCT year FROM songs WHERE year IS NOT NULL ORDER BY year DESC")->fetchAll(PDO::FETCH_COLUMN);
    $languages = $pdo->query("SELECT DISTINCT language FROM songs WHERE language != '' ORDER BY language")->fetchAll(PDO::FETCH_COLUMN);
    $albums = $pdo->query("SELECT DISTINCT album FROM songs WHERE album != '' ORDER BY album")->fetchAll(PDO::FETCH_COLUMN);

    wp_send_json_success([
        'songs' => $songs,
        'pagination' => [
            'current_page' => $page,
            'total_songs' => $total_songs,
            'songs_per_page' => $songs_per_page,
            'total_pages' => ceil($total_songs / $songs_per_page)
        ],
        'filters' => [
            'genres' => $genres,
            'years' => $years,
            'languages' => $languages,
            'albums' => $albums
        ]
    ]);
}


add_action('wp_ajax_get_song_details', 'song_manager_ajax_get_song_details');
add_action('wp_ajax_nopriv_get_song_details', 'song_manager_ajax_get_song_details');

function song_manager_ajax_get_song_details() {
    if (!wp_verify_nonce($_POST['nonce'], 'song_manager_nonce')) {
        wp_die('Security check failed');
    }
    
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        wp_send_json_error('Database connection failed');
    }
    
    $song_id = intval($_POST['song_id']);
    
    $stmt = $pdo->prepare("SELECT * FROM songs WHERE id = ?");
    $stmt->execute([$song_id]);
    $song = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$song) {
        wp_send_json_error('Song not found');
    }
    
    wp_send_json_success($song);
}

function song_manager_enqueue_scripts($hook) {
    if (is_admin()) {
        if (strpos($hook, 'song-manager') === false && strpos($hook, 'song-manager-') === false && strpos($hook, 'toplevel_page_song-manager') === false) {
            wp_enqueue_style('music', plugin_dir_url(__FILE__) . 'music.css');
            return;
        }
    }
    wp_enqueue_script('jquery');
    wp_enqueue_media();
    wp_enqueue_style('music', plugin_dir_url(__FILE__) . 'music.css');
    wp_enqueue_script('music-admin',plugin_dir_url(__FILE__) . 'music-admin.js',array('jquery'),'1.0',true);
    wp_localize_script('music-admin', 'musicUploader', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce'   => wp_create_nonce('music_upload_nonce')
    ));
    $ajax_url = admin_url('admin-ajax.php');
    $nonce = wp_create_nonce('song_manager_nonce');

    $inline_js = <<<JS
    jQuery(document).ready(function($){
        // Build filter behavior
        function filterAlbums() {
            var q = $('#album-search').val().toLowerCase();
            $('#album-container .album-item').each(function(){
                var name = $(this).data('title').toLowerCase();
                var artist = $(this).data('artist') ? $(this).data('artist').toLowerCase() : '';
                if (name.indexOf(q) !== -1 || artist.indexOf(q) !== -1 || q === '') {
                    $(this).show();
                } else {
                    $(this).hide();
                }
            });
        }

    $(document).on('input', '#album-search', function(){
        filterAlbums();
    });

    // Click album item -> select it, set hidden album_id, overwrite fields and preview cover
    $(document).on('click', '.album-item', function(e){
        e.preventDefault();
        var \$item = $(this);
        $('#album-container .album-item').removeClass('selected');
        \$item.addClass('selected');

        var albumId = \$item.data('id');
        var title = \$item.data('title');
        var artist = \$item.data('artist');
        var genre = \$item.data('genre');
        var year = \$item.data('year');
        var language = \$item.data('language');
        var cover = \$item.data('cover');

        // set hidden album_id
        $('#album_id').val(albumId);

        // overwrite text album field (option A)
        $('input[name=\"album\"]').val(title);

        // overwrite other song fields if they are empty OR always overwrite? We will overwrite as requested.
        $('input[name=\"artist\"]').val(artist);
        $('input[name=\"genre\"]').val(genre);
        $('input[name=\"year\"]').val(year);
        $('input[name=\"language\"]').val(language);

        // cover input and preview
        $('#cover').val(cover || '');
        if (cover) {
            $('#cover_preview').html('<img src=\"'+cover+'\" style=\"max-width:100px; height:auto;\">');
        } else {
            $('#cover_preview').html('');
        }
    });

    // Optional: double-click to open album details in new tab (if you want)
    // $(document).on('dblclick', '.album-item', function(){ window.open($(this).data('link') || '#'); });

    // If album_id already set (editing), visually mark it
    var currentAlbumId = $('#album_id').val();
    if (currentAlbumId) {
        $('#album-container .album-item').each(function(){
            if ($(this).data('id') == currentAlbumId) {
                $(this).addClass('selected');
            }
        });
    }
    });
    JS;

    wp_add_inline_script('jquery', $inline_js);
}
add_action('admin_enqueue_scripts', 'song_manager_enqueue_scripts', 10, 1);
function song_manager_enqueue_frontend_assets() {
    wp_enqueue_style('music', plugin_dir_url(__FILE__) . 'music.css');
    if (is_singular() && (has_shortcode(get_post()->post_content, 'music') || has_shortcode(get_post()->post_content, 'songs'))) {
        wp_enqueue_style(
            'music',
            plugin_dir_url(__FILE__) . 'music.css',
            [],
            '1.0'
        );
        wp_enqueue_script('jquery');
    }
}
add_action('wp_enqueue_scripts', 'song_manager_enqueue_frontend_assets');

/**
 * Admin Media uploader (for both songs and albums)
 */
function song_manager_admin_media_uploader($hook) {
    // Only load on our plugin admin pages (page slugs contain 'song-manager')
    if (strpos($hook, 'song-manager') === false) {
        return;
    }

    wp_enqueue_media();
    wp_enqueue_script('jquery');

    $inline_js = "
        jQuery(document).ready(function($){
            var frame;
            $('#upload_cover_button').on('click', function(e){
                e.preventDefault();
                if (typeof wp === 'undefined' || !wp.media) {
                    alert('Media library is not available.');
                    return;
                }
                frame = wp.media({
                    title: 'Select or Upload Cover Image',
                    button: { text: 'Use this image' },
                    multiple: false
                });
                frame.on('select', function(){
                    var attachment = frame.state().get('selection').first().toJSON();
                    $('#cover').val(attachment.url);
                    $('#cover_preview').html('<img src=\"'+attachment.url+'\" style=\"max-width:100px; height:auto;\">');
                });
                frame.open();
            });

            var albumFrame;
            $('#upload_album_cover_button').on('click', function(e){
                e.preventDefault();
                if (typeof wp === 'undefined' || !wp.media) {
                    alert('Media library is not available.');
                    return;
                }
                albumFrame = wp.media({
                    title: 'Select or Upload Album Cover',
                    button: { text: 'Use this image' },
                    multiple: false
                });
                albumFrame.on('select', function(){
                    var attachment = albumFrame.state().get('selection').first().toJSON();
                    $('#album_cover').val(attachment.url);
                    $('#album_cover_preview').html('<img src=\"'+attachment.url+'\" style=\"max-width:100px; height:auto;\">');
                });
                albumFrame.open();
            });
        });
    ";
    wp_add_inline_script('jquery', $inline_js);
}
add_action('admin_enqueue_scripts', 'song_manager_admin_media_uploader');

add_action('admin_footer', 'song_manager_admin_footer');
function song_manager_admin_footer() {
    $screen = get_current_screen();
    if ($screen && $screen->id === 'toplevel_page_song-manager') {
        ?>
        <script>
        jQuery(document).ready(function($) {
            $('a[href*="action=deletesong"]').on('click', function(e) {
                if (!confirm('Are you sure you want to delete this item?')) {
                    e.preventDefault();
                    return false;
                }
            });
        });
        </script>
        <?php
    }
}

add_action('wp_ajax_get_album_details', 'song_manager_ajax_get_album_details');
add_action('wp_ajax_nopriv_get_album_details', 'song_manager_ajax_get_album_details');

function song_manager_ajax_get_album_details() {
    // security check (optional - you can require the same nonce as other endpoints)
    if (isset($_POST['nonce']) && !wp_verify_nonce($_POST['nonce'], 'song_manager_nonce')) {
        wp_send_json_error('Security check failed');
    }

    if (!isset($_POST['album_id'])) {
        wp_send_json_error('Missing album_id');
    }

    $album_id = intval($_POST['album_id']);
    $pdo = song_manager_get_db_connection();
    if (!$pdo) {
        wp_send_json_error('Database connection failed');
    }

    try {
        $stmt = $pdo->prepare("SELECT id, title, artist, cover, genre, year, language FROM albums WHERE id = ?");
        $stmt->execute([$album_id]);
        $album = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$album) {
            wp_send_json_error('Album not found');
        }
        wp_send_json_success($album);
    } catch (Exception $e) {
        wp_send_json_error('Error: ' . $e->getMessage());
    }
}

function sp_get_song_by_id($id) {
    global $wpdb;
    $table_name = 'songs';
    return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %s", $id));
}

function sp_single_song_shortcode($atts) {
    global $wpdb;
    $atts = shortcode_atts(['song' => ''], $atts);

    $song_id = intval($atts['song']);
    if ($song_id <= 0) {
        return '<p>Invalid song ID.</p>';
    }

    $song = $wpdb->get_row(
        $wpdb->prepare("SELECT * FROM songs WHERE id = %d", $song_id)
    );

    if (!$song) {
        return '<p>Song not found.</p>';
    }

    ob_start(); ?>
    <div class="sp-single-line">
        <div class="sp-title"><?php echo esc_html($song->title); ?></div>
        <audio controls class="sp-audio">
            <source src="<?php echo esc_url($song->audio_file); ?>" type="audio/mpeg">
            Your browser does not support the audio element.
        </audio>
        <?php if (!empty($song->cover)): ?>
            <img src="<?php echo esc_url($song->cover); ?>" alt="<?php echo esc_attr($song->title); ?>" class="sp-cover">
        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('player', 'sp_single_song_shortcode');





require_once plugin_dir_path(__FILE__) . 'update.php';
if (class_exists('music_Plugin_Updater')) {
    new music_Plugin_Updater(__FILE__, 'music', 'https://it-breeze.cloud/data/music/music.json');
}
