<?php

if (!defined('ABSPATH')) {
    exit;
}

class music_Plugin_Updater {
    private $plugin_file;
    private $plugin_slug;
    private $plugin_folder;
    private $remote_url;
    
    public function __construct($plugin_file, $plugin_slug, $remote_url) {
        $this->plugin_file = $plugin_file;
        $this->plugin_slug = $plugin_slug;
        $this->plugin_folder = dirname(plugin_basename($plugin_file));
        $this->remote_url = $remote_url;
        
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_update'));
        add_filter('plugins_api', array($this, 'plugin_info'), 20, 3);
        add_filter('plugin_row_meta', array($this, 'add_view_details_link'), 10, 4);
    }
    
    public function check_for_update($checked_data) {
        if (empty($checked_data->checked)) {
            return $checked_data;
        }
        
        $plugin_basename = plugin_basename($this->plugin_file);
        
        if (!isset($checked_data->checked[$plugin_basename])) {
            return $checked_data;
        }
        
        $plugin_data = get_plugin_data($this->plugin_file);
        $plugin_version = $plugin_data['Version'];
        $response = wp_remote_get($this->remote_url);
        
        if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {
            return $checked_data;
        }
        
        $remote_data = json_decode(wp_remote_retrieve_body($response));
        
        if (version_compare($plugin_version, $remote_data->new_version, '<')) {
            $checked_data->response[$plugin_basename] = (object) array(
                'slug' => $this->plugin_folder,
                'plugin' => $plugin_basename,
                'new_version' => $remote_data->new_version,
                'package' => $remote_data->package,
                'url' => $remote_data->url,
            );
        }
        
        return $checked_data;
    }
    
    public function plugin_info($false, $action, $args) {
        if ($action !== 'plugin_information') {
            return $false;
        }

        if (empty($args->slug) || $args->slug !== $this->plugin_slug) {
            return $false;
        }

        $response = wp_remote_get($this->remote_url);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return $false;
        }

        $remote_data = json_decode(wp_remote_retrieve_body($response));

        if (empty($remote_data)) {
            return $false;
        }

        return (object) [
            'name'           => $remote_data->name,
            'slug'           => $this->plugin_slug,
            'version'        => $remote_data->new_version,
            'author'         => $remote_data->author,
            'download_link'  => $remote_data->package,
            'requires'       => $remote_data->requires,
            'tested'         => $remote_data->tested,
            'requires_php'   => $remote_data->requires_php,
            'last_updated'   => $remote_data->last_updated,
            'homepage'       => $remote_data->homepage,
            'sections'       => [
                'description'  => $remote_data->description,
                'installation' => $remote_data->installation,
                'faq'          => $remote_data->faq,
                'changelog'    => $remote_data->changelog,
            ],
            'plugin'         => plugin_basename($this->plugin_file),
            'banners'        => [],
            'icons'          => [],
            'ratings'        => $remote_data->ratings,
            'rating'         => $remote_data->rating,
            'num_ratings'    => $remote_data->num_ratings,
            'active_installs'=> $remote_data->active_installs,
        ];
    }
    
    public function add_view_details_link($plugin_meta, $plugin_file, $plugin_data, $status) {
        if ($plugin_file === plugin_basename($this->plugin_file)) {
            $update_plugins = get_site_transient('update_plugins');
            $has_update = isset($update_plugins->response[$plugin_file]);
            
            if (!$has_update) {
                $details_url = add_query_arg(
                    array(
                        'tab' => 'plugin-information',
                        'plugin' => $this->plugin_folder,
                        'TB_iframe' => 'true',
                        'width' => '772',
                        'height' => '949'
                    ),
                    admin_url('plugin-install.php')
                );
                
                $plugin_meta[] = sprintf(
                    '<a href="%s" class="thickbox open-plugin-details-modal" aria-label="%s" data-title="%s">%s</a>',
                    esc_url($details_url),
                    esc_attr(sprintf(__('More information about %s'), $plugin_data['Name'])),
                    esc_attr($plugin_data['Name']),
                    __('View details')
                );
            }
        }
        
        return $plugin_meta;
    }
}
