<?php
/**
 * Plugin Name: News Carousel
 * Plugin URI: https://it-breeze.info
 * Description: Manage news and display them in a carousel or marquee using the shortcodes [news] and [news section="marquee"].
 * Version: 2.2.0
 * Author: Mike Vahldieck
 * Author URI: https://it-breeze.info
 * License: GPL v2 or later
 */

if (! defined('ABSPATH')) {
    exit;
}

/* -------------------------
   Register 'news' custom post type
   ------------------------- */
function nc_register_news_post_type() {
    $labels = array(
        'name' => 'News',
        'singular_name' => 'News',
        'menu_name' => 'News',
    );
    $args = array(
        'labels' => $labels,
        'public' => true,
        'has_archive' => true,
        'show_in_menu' => true,
        'menu_icon' => 'dashicons-media-document',
        'supports' => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
        'menu_position' => 5,
        'rewrite' => array('slug' => 'news', 'with_front' => false),
    );
    register_post_type('news', $args);
}
add_action('init', 'nc_register_news_post_type');

/* -------------------------
   Meta boxes (start/end + active)
   ------------------------- */
function nc_add_news_meta_boxes() {
    add_meta_box('nc_news_dates', 'News Dates and Status', 'nc_news_dates_callback', 'news', 'side', 'default');
}
add_action('add_meta_boxes', 'nc_add_news_meta_boxes');

function nc_news_dates_callback($post) {
    wp_nonce_field('nc_news_dates_save', 'nc_news_dates_nonce');
    $start = get_post_meta($post->ID, '_nc_news_start_date', true);
    $end = get_post_meta($post->ID, '_nc_news_end_date', true);
    $active = get_post_meta($post->ID, '_nc_news_active', true);
    $show_read_more = get_post_meta($post->ID, '_nc_news_show_read_more', true);
    ?>
    <p>
        <label for="nc_news_start_date">Start Date:</label><br/>
        <input type="date" id="nc_news_start_date" name="nc_news_start_date" value="<?php echo esc_attr($start); ?>" />
    </p>
    <p>
        <label for="nc_news_end_date">End Date:</label><br/>
        <input type="date" id="nc_news_end_date" name="nc_news_end_date" value="<?php echo esc_attr($end); ?>" />
    </p>
    <p>
        <label>
            <input type="checkbox" id="nc_news_show_read_more" name="nc_news_show_read_more" value="yes" <?php checked($show_read_more, 'yes'); ?> />
            Show "Read More" button
        </label>
    </p>
    <p>
        <label>
            <input type="checkbox" id="nc_news_active" name="nc_news_active" value="yes" <?php checked($active, 'yes'); ?> />
            Active
        </label>
    </p>
    <?php
}

function nc_save_news_meta($post_id) {
    if (!isset($_POST['nc_news_dates_nonce']) || !wp_verify_nonce($_POST['nc_news_dates_nonce'], 'nc_news_dates_save')) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $fields = [
        'nc_news_start_date' => '_nc_news_start_date',
        'nc_news_end_date'   => '_nc_news_end_date',
    ];

    foreach ($fields as $field => $meta_key) {
        if (isset($_POST[$field]) && $_POST[$field] !== '') {
            update_post_meta($post_id, $meta_key, sanitize_text_field($_POST[$field]));
        } else {
            delete_post_meta($post_id, $meta_key);
        }
    }

    $active = (isset($_POST['nc_news_active']) && $_POST['nc_news_active'] === 'yes') ? 'yes' : 'no';
    update_post_meta($post_id, '_nc_news_active', $active);

    $show_read_more = (isset($_POST['nc_news_show_read_more']) && $_POST['nc_news_show_read_more'] === 'yes') ? 'yes' : 'no';
    update_post_meta($post_id, '_nc_news_show_read_more', $show_read_more);
}
add_action('save_post', 'nc_save_news_meta');

/* -------------------------
   Settings registration & page
   ------------------------- */
// (same as before — no licensing code here)

/* -------------------------
   Settings registration & page
   ------------------------- */
function nc_register_news_settings() {
    // Marquee & default view
    add_option('nc_marquee_divider','|');
    add_option('nc_marquee_speed',50); // seconds (higher = slower)
    add_option('nc_default_speed',5000);
    add_option('nc_default_controls','dots');

    register_setting('nc_news_settings_group','nc_marquee_divider','sanitize_text_field');
    register_setting('nc_news_settings_group','nc_marquee_speed','intval');
    register_setting('nc_news_settings_group','nc_default_speed','intval');
    register_setting('nc_news_settings_group','nc_default_controls', function($v){
        $valid = array('dots','arrows','both');
        return in_array($v,$valid) ? $v : 'dots';
    });

    // Display settings (new)
    add_option('nc_featured_width',300);
    add_option('nc_featured_width_unit','px');
    add_option('nc_header_tag','h3');
    add_option('nc_text_size','16px'); // free-form CSS unit
    add_option('nc_bg_color','#ffffff');
    add_option('nc_text_color','#000000');

    register_setting('nc_news_settings_group','nc_featured_width','intval');
    register_setting('nc_news_settings_group','nc_featured_width_unit', function($v){
        return ($v === 'rem') ? 'rem' : 'px';
    });
    register_setting('nc_news_settings_group','nc_header_tag', function($v){
        $valid = array('h2','h3','h4','h5','h6');
        return in_array($v,$valid) ? $v : 'h3';
    });
    register_setting('nc_news_settings_group','nc_text_size','sanitize_text_field');
    register_setting('nc_news_settings_group','nc_bg_color','sanitize_hex_color');
    register_setting('nc_news_settings_group','nc_text_color','sanitize_hex_color');
}
add_action('admin_init','nc_register_news_settings');

function nc_register_news_settings_page() {
    add_submenu_page(
        'edit.php?post_type=news',
        'News Settings',
        'Settings',
        'manage_options',
        'nc_news_settings',
        'nc_news_settings_page_html'
    );
}
add_action('admin_menu','nc_register_news_settings_page');

function nc_news_settings_page_html() {
    if (! current_user_can('manage_options')) return;

    $featured_width = intval(get_option('nc_featured_width',300));
    $featured_width_unit = get_option('nc_featured_width_unit','px');
    $header_tag = get_option('nc_header_tag','h3');
    $text_size = get_option('nc_text_size','16px');
    $bg_color = get_option('nc_bg_color','#ffffff');
    $text_color = get_option('nc_text_color','#000000');
    ?>
    <div class="wrap">
        <h1>News Settings</h1>
        <form method="post" action="options.php">
            <?php settings_fields('nc_news_settings_group'); ?>

            <h2>Marquee</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">Divider</th>
                    <td><input type="text" name="nc_marquee_divider" value="<?php echo esc_attr(get_option('nc_marquee_divider','|')); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row">Scroll Duration</th>
                    <td><input type="number" name="nc_marquee_speed" value="<?php echo esc_attr(get_option('nc_marquee_speed',50)); ?>" /> seconds (higher = slower)</td>
                </tr>
            </table>

            <h2>Default View (carousel)</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">Scroll Speed (ms)</th>
                    <td><input type="number" name="nc_default_speed" value="<?php echo esc_attr(get_option('nc_default_speed',5000)); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row">Controls</th>
                    <td>
                        <select name="nc_default_controls">
                            <option value="dots" <?php selected(get_option('nc_default_controls'),'dots'); ?>>Dots</option>
                            <option value="arrows" <?php selected(get_option('nc_default_controls'),'arrows'); ?>>Arrows</option>
                            <option value="both" <?php selected(get_option('nc_default_controls'),'both'); ?>>Both</option>
                        </select>
                    </td>
                </tr>
            </table>

            <h2>Display Settings</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">Featured Image Width</th>
                    <td>
                        <div id="nc_featured_width_slider" style="width:320px;margin-bottom:8px;"></div>
                        <input type="hidden" id="nc_featured_width" name="nc_featured_width" value="<?php echo esc_attr($featured_width); ?>" />
                        <input type="number" id="nc_featured_width_number" min="50" max="800" step="1" value="<?php echo esc_attr($featured_width); ?>" style="width:80px;margin-left:8px;" />
                        <select id="nc_featured_width_unit" name="nc_featured_width_unit" style="margin-left:8px;">
                            <option value="px" <?php selected($featured_width_unit,'px'); ?>>px</option>
                            <option value="rem" <?php selected($featured_width_unit,'rem'); ?>>rem</option>
                        </select>
                        <p class="description">Adjust featured image width. Height will auto-adjust to preserve aspect ratio.</p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">Header Size</th>
                    <td>
                        <select name="nc_header_tag">
                            <option value="h2" <?php selected($header_tag,'h2'); ?>>h2</option>
                            <option value="h3" <?php selected($header_tag,'h3'); ?>>h3</option>
                            <option value="h4" <?php selected($header_tag,'h4'); ?>>h4</option>
                            <option value="h5" <?php selected($header_tag,'h5'); ?>>h5</option>
                            <option value="h6" <?php selected($header_tag,'h6'); ?>>h6</option>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th scope="row">Text Size</th>
                    <td>
                        <input type="text" id="nc_text_size" name="nc_text_size" value="<?php echo esc_attr($text_size); ?>" placeholder="e.g. 16px or 1.2rem" style="width:120px;" />
                        <p class="description">Enter any valid CSS size (e.g. <code>16px</code>, <code>1rem</code>, <code>120%</code>).</p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">Background Color</th>
                    <td><input class="nc-color-field" type="text" id="nc_bg_color" name="nc_bg_color" value="<?php echo esc_attr($bg_color); ?>" /></td>
                </tr>

                <tr>
                    <th scope="row">Text Color</th>
                    <td><input class="nc-color-field" type="text" id="nc_text_color" name="nc_text_color" value="<?php echo esc_attr($text_color); ?>" /></td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>

    <style>
    /* minimal admin slider styling fallback if theme doesn't provide jQuery UI CSS */
    #nc_featured_width_slider { margin-top:6px; }
    </style>

    <?php
}

/* -------------------------
   Admin scripts/styles for settings page
   ------------------------- */
function nc_admin_enqueue($hook_suffix) {
    if (! isset($_GET['page']) || $_GET['page'] !== 'nc_news_settings') return;

    // Color picker
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');

    // jQuery UI slider
    wp_enqueue_script('jquery-ui-slider');
    // optional jQuery UI CSS from CDN (fallback). If your environment blocks external assets, the slider still works but looks unstyled.
    wp_enqueue_style('nc-jquery-ui-css', 'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css', array(), null);

    $featured_width = intval(get_option('nc_featured_width',300));
    $inline = "
    jQuery(document).ready(function($){
        $('.nc-color-field').wpColorPicker();

        $('#nc_featured_width_slider').slider({
            min: 50,
            max: 800,
            step: 1,
            value: {$featured_width},
            slide: function(event, ui){
                $('#nc_featured_width').val(ui.value);
                $('#nc_featured_width_number').val(ui.value);
            }
        });

        $('#nc_featured_width_number').on('input', function(){
            var v = parseInt($(this).val(),10) || {$featured_width};
            v = Math.max(50, Math.min(800, v));
            $('#nc_featured_width').val(v);
            $('#nc_featured_width_slider').slider('value', v);
        });
    });
    ";
    wp_add_inline_script('jquery-ui-slider',$inline);
}
add_action('admin_enqueue_scripts','nc_admin_enqueue');

/* -------------------------
   Frontend assets (centralized inline)
   ------------------------- */
function nc_enqueue_assets() {
    // Register an inline-only script correctly so WP doesn't print an empty src attribute
    wp_register_script('nc-frontend', false, array('jquery'), false, true);

    // Localized settings for JS/AJAX
    $settings = array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'ajax_nonce' => wp_create_nonce('nc_load_news_nonce'),
        'default_speed' => intval(get_option('nc_default_speed',5000)),
        'default_controls' => get_option('nc_default_controls','dots'),
        'marquee_speed' => intval(get_option('nc_marquee_speed',50)),
    );
    wp_localize_script('nc-frontend','ncSettings',$settings);

    // Inline JS (clean, no syntax errors)
    $js = <<<'JS'
(function($){
    function ncAjaxLoad(post_id){
        if(!post_id) return;
        $.post(ncSettings.ajax_url, { action: 'nc_load_news', post_id: post_id, _nc_nonce: ncSettings.ajax_nonce }, function(response){
            if(response && response.success && response.data){
                var $main = $('#main');
                if($main.length){
                    $main.html(response.data);
                    $('html,body').animate({ scrollTop: $main.offset().top }, 300);
                } else {
                    $('body').append(response.data);
                }
            }
        }, 'json');
    }

    function initCarousel(root, interval, controls) {
        var $root = $(root);
        var $carousel = $root.find('.nc-carousel-inner');
        var $items = $carousel.children('.nc-carousel-item');
        if ($items.length < 2) return; // nothing to rotate

        var currentIndex = 0;
        var carouselInterval = null;
        var $dots = $root.find('.nc-carousel-dots');

        function showItem(idx){
            $items.removeClass('active').eq(idx).addClass('active');
            if ($dots.length && (controls === 'dots' || controls === 'both')) {
                $dots.find('span').removeClass('active').eq(idx).addClass('active');
            }
        }

        // Build dots if requested
        if ((controls === 'dots' || controls === 'both') && $dots.length) {
            $dots.empty();
            $items.each(function(i){
                $dots.append($('<span>').attr('data-index', i));
            });
            $dots.off('click').on('click', 'span', function(){
                var i = parseInt($(this).attr('data-index'), 10) || 0;
                currentIndex = i;
                showItem(currentIndex);
                restart();
            });
        }

        // Arrows handling
        if (controls === 'arrows' || controls === 'both') {
            $root.off('click', '.nc-carousel-prev').on('click', '.nc-carousel-prev', function(e){
                e.preventDefault();
                currentIndex = (currentIndex - 1 + $items.length) % $items.length;
                showItem(currentIndex);
                restart();
            });
            $root.off('click', '.nc-carousel-next').on('click', '.nc-carousel-next', function(e){
                e.preventDefault();
                currentIndex = (currentIndex + 1) % $items.length;
                showItem(currentIndex);
                restart();
            });
        }

        function start(){
            stop();
            carouselInterval = setInterval(function(){
                currentIndex = (currentIndex + 1) % $items.length;
                showItem(currentIndex);
            }, interval);
        }
        function stop(){
            if (carouselInterval) {
                clearInterval(carouselInterval);
                carouselInterval = null;
            }
        }
        function restart(){
            stop();
            start();
        }

        $carousel.off('mouseenter mouseleave').on('mouseenter', stop).on('mouseleave', start);
        showItem(currentIndex);
        start();
    }

    function initMarquee(root) {
        var $root = $(root);
        var $inner = $root.find('.nc-marquee-inner');
        if (!$inner.length) return;
        $inner.off('click', '.nc-marquee-item').on('click', '.nc-marquee-item', function(e){
            e.preventDefault();
            var id = $(this).data('post-id');
            if (id) ncAjaxLoad(id);
        });
    }

    $(function(){
        $('.nc-news-root').each(function(){
            var $root = $(this);
            var mode = $root.data('nc-mode') || 'carousel';
            if (mode === 'marquee') {
                initMarquee(this);
            } else {
                var interval = parseInt($root.data('nc-interval'), 10) || (window.ncSettings ? parseInt(ncSettings.default_speed,10) : 5000);
                var controls = $root.data('nc-controls') || (window.ncSettings ? ncSettings.default_controls : 'dots');
                initCarousel(this, interval, controls);

                // Read-more handling (delegate)
                $root.off('click', '.nc-read-more').on('click', '.nc-read-more', function(e){
                    e.preventDefault();
                    var id = $(this).data('post-id') || $(this).closest('.nc-carousel-item').data('post-id');
                    if (id) ncAjaxLoad(id);
                });
            }
        });
    });
})(jQuery);
JS;

    wp_add_inline_script('nc-frontend', $js);
    wp_enqueue_script('nc-frontend');

    // CSS - include marquee animation using existing setting (seconds)
    $marquee_speed = intval(get_option('nc_marquee_speed',50));
    if ($marquee_speed < 5) $marquee_speed = 5; // sane minimum

    $css = "
    .nc-carousel { position: relative; }
    .nc-carousel-inner { position: relative; width: 100%; overflow: hidden; }
    .nc-carousel-item { display:none; width:100%; margin:0 auto; }
    .nc-carousel-item.active { display:block; }
    .nc-carousel-card { border:none; padding:10px; margin-top:-16px; box-shadow:0 2px 4px rgba(0,0,0,0.1); border-radius:6px; }
    .nc-carousel-image { margin-bottom:15px; text-align:center; }
    .nc-carousel-content { text-align:center; }
    .nc-carousel-excerpt { display:-webkit-box; -webkit-line-clamp:8; -webkit-box-orient:vertical; overflow:hidden; text-overflow:ellipsis; max-height:calc(1.5em * 8); margin-bottom:15px; }
    .nc-read-more { display:inline-block; margin-top:15px; padding:10px 20px; text-decoration:none; border:1px solid grey; border-radius:66px; cursor:pointer; }
    .nc-read-more:hover { background:#e8e8e8; color:#000; }
    .nc-carousel-dots { text-align:center; margin-top:10px; }
    .nc-carousel-dots span { display:inline-block; width:10px; height:10px; background:#ccc; margin:0 5px; border-radius:50%; cursor:pointer; }
    .nc-carousel-dots span.active { background:#d3d3d3; }
    .nc-carousel-arrow { position:absolute; top:50%; transform:translateY(-50%); font-size:24px; cursor:pointer; color:#333; user-select:none; padding:5px 10px; background:rgba(255,255,255,0.7); border-radius:50%; }
    .nc-carousel-prev { left:10px; }
    .nc-carousel-next { right:10px; }
    .nc-centered-thumbnail img { display:block; margin:0 auto; height:auto; }
    .nc-centered-thumbnail { display:block; margin:0 auto; }

    /* Marquee: inner duplicated strip animated with CSS */
    .nc-news-marquee { overflow:hidden; white-space:nowrap; position:relative; }
    .nc-marquee-inner { display:inline-block; white-space:nowrap; animation: nc-marquee-scroll " . esc_html($marquee_speed) . "s linear infinite; }
    @keyframes nc-marquee-scroll {
        0% { transform: translateX(0); }
        100% { transform: translateX(-50%); }
    }
    .nc-marquee-item { display:inline-block; margin:0 10px; cursor:pointer; }
    .nc-marquee-divider { margin:0 10px; display:inline-block; vertical-align:middle; }
    ";
    wp_register_style('nc-frontend-style', false);
    wp_add_inline_style('nc-frontend-style', $css);
    wp_enqueue_style('nc-frontend-style');
}
add_action('wp_enqueue_scripts','nc_enqueue_assets');

/* -------------------------
   Shortcode: [news] and [news section="marquee"]
   ------------------------- */
function nc_build_news_query() {
    // Respect WordPress timezone to avoid day-boundary issues
    $today = current_time('Y-m-d');

    return array(
        'post_type'      => 'news',
        'posts_per_page' => -1,
        'meta_query'     => array(
            'relation' => 'AND',
            // Accept both legacy 'yes' and newer '1' (plus common truthy variants)
            array(
                'key'     => '_nc_news_active',
                'value'   => array('yes','1','true','on'),
                'compare' => 'IN',
            ),
            array(
                'key'     => '_nc_news_start_date',
                'value'   => $today,
                'compare' => '<=',
                'type'    => 'DATE',
            ),
            array(
                'key'     => '_nc_news_end_date',
                'value'   => $today,
                'compare' => '>=',
                'type'    => 'DATE',
            ),
        ),
    );
}


function nc_news_shortcode($atts = array()) {
    $atts = shortcode_atts(array('section' => 'default'), $atts, 'news');
    $args = nc_build_news_query();
    $query = new WP_Query($args);
    ob_start();

    if (! $query->have_posts()) {
        echo '<h5>No news available.</h5>';
        wp_reset_postdata();
        return ob_get_clean();
    }

    // display settings
    $featured_width = intval(get_option('nc_featured_width',300));
    $featured_unit = get_option('nc_featured_width_unit','px');
    $header_tag = get_option('nc_header_tag','h3');
    $text_size = get_option('nc_text_size','16px');
    $bg_color = get_option('nc_bg_color','#ffffff');
    $text_color = get_option('nc_text_color','#000000');

    $interval = intval(get_option('nc_default_speed',5000));
    $controls = get_option('nc_default_controls','dots');
    $controls = in_array($controls,array('dots','arrows','both')) ? $controls : 'dots';
    $marquee_divider = esc_html(get_option('nc_marquee_divider','|'));
    $marquee_speed = intval(get_option('nc_marquee_speed',50));

    $wrapper_style = 'background-color:' . esc_attr($bg_color) . '; color:' . esc_attr($text_color) . ';';
    $title_style = 'font-size:' . esc_attr($text_size) . '; margin:.5em 0;';
    $text_style = 'font-size:' . esc_attr($text_size) . ';';

    if ($atts['section'] === 'marquee') {
        $items = array();
        while ($query->have_posts()) : $query->the_post();
            $items[] = '<span class="nc-marquee-item" data-post-id="' . esc_attr(get_the_ID()) . '">' . esc_html(get_the_title()) . '</span>';
        endwhile;
        $sequence = implode('<span class="nc-marquee-divider">' . $marquee_divider . '</span>', $items);
        $duplicated = $sequence . '<span class="nc-marquee-divider">' . $marquee_divider . '</span>' . $sequence;

        echo '<div class="nc-news-root" data-nc-mode="marquee" data-nc-marquee-speed="' . esc_attr($marquee_speed) . '">';
        echo '<div class="nc-news-marquee"><div class="nc-marquee-inner">' . $duplicated . '</div></div>';
        echo '</div>';
    } else {
        echo '<div class="nc-news-root" data-nc-mode="carousel" data-nc-interval="' . esc_attr($interval) . '" data-nc-controls="' . esc_attr($controls) . '">';
        echo '<div class="nc-carousel"><div class="nc-carousel-inner">';

        $first_item = true;
        while ($query->have_posts()) : $query->the_post();
            $post_id = get_the_ID();

            // render carousel item; first item gets 'active' so it's visible before JS
            echo '<div class="nc-carousel-item' . ($first_item ? ' active' : '') . '" data-post-id="' . esc_attr($post_id) . '">';
            echo '<div class="nc-carousel-card" style="' . $wrapper_style . '">';
            echo '<div class="nc-carousel-content">';

            if (has_post_thumbnail()) {
                $img_html = get_the_post_thumbnail($post_id, 'full', array('style' => 'width:' . esc_attr($featured_width) . esc_attr($featured_unit) . '; height:auto; display:block; margin:0 auto;'));
                echo '<div class="nc-carousel-image">' . $img_html . '</div>';
            }

            echo '<' . esc_attr($header_tag) . ' class="nc-news-title" style="' . $title_style . '">' . esc_html(get_the_title()) . '</' . esc_attr($header_tag) . '>';
            $show_read_more = get_post_meta($post_id, '_nc_news_show_read_more', true);

            if ($show_read_more === 'yes') {
                // Show excerpt + button
                echo '<div class="nc-carousel-excerpt" style="' . $text_style . '">' . wp_kses_post(get_the_excerpt()) . '</div>';
                echo '<a href="#" class="nc-read-more" data-post-id="' . esc_attr($post_id) . '">Read More</a>';
            } else {
                // Show full content (with filters so shortcodes/blocks render correctly)
                echo '<div class="nc-carousel-content-full" style="' . $text_style . '">' . wp_kses_post(apply_filters('the_content', get_the_content())) . '</div>';
            }
            echo '</div></div></div>';
            $first_item = false;
        endwhile;

        echo '</div>'; // .nc-carousel-inner

        if ($controls === 'dots' || $controls === 'both') echo '<div class="nc-carousel-dots"></div>';
        if ($controls === 'arrows' || $controls === 'both') {
            echo '<div class="nc-carousel-arrow nc-carousel-prev" role="button" aria-label="prev">&#10094;</div>';
            echo '<div class="nc-carousel-arrow nc-carousel-next" role="button" aria-label="next">&#10095;</div>';
        }

        echo '</div></div>'; // carousel, root
    }

    wp_reset_postdata();
    return ob_get_clean();
}
add_shortcode('news','nc_news_shortcode');

/* -------------------------
   AJAX handler for Read More
   ------------------------- */
add_action('wp_ajax_nc_load_news','nc_load_news_ajax');
add_action('wp_ajax_nopriv_nc_load_news','nc_load_news_ajax');

function nc_load_news_ajax() {
    $nonce = isset($_REQUEST['_nc_nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['_nc_nonce'])) : '';
    if (! wp_verify_nonce($nonce,'nc_load_news_nonce')) {
        wp_send_json_error('Invalid nonce.',403);
    }

    if (! isset($_POST['post_id'])) wp_send_json_error('No post ID.');
    $post_id = intval($_POST['post_id']);
    $post = get_post($post_id);
    if (! $post || $post->post_type !== 'news') wp_send_json_error('Invalid post.');

    // check active & dates
    $active = get_post_meta($post_id,'_nc_news_active',true);
    $start = get_post_meta($post_id,'_nc_news_start_date',true);
    $end = get_post_meta($post_id,'_nc_news_end_date',true);
    $today = date('Y-m-d');
    if ($active !== 'yes') wp_send_json_error('Not active.');
    if ($start && $start > $today) wp_send_json_error('Not started.');
    if ($end && $end < $today) wp_send_json_error('Expired.');

    // apply featured image width setting to AJAX content too
    $featured_width = intval(get_option('nc_featured_width',300));
    $featured_unit = get_option('nc_featured_width_unit','px');

    $featured_html = has_post_thumbnail($post->ID)
        ? '<div class="nc-centered-thumbnail" style="width:100%;display:flex;justify-content:center;">'
            . get_the_post_thumbnail($post->ID, 'large', array('style' => 'width:' . esc_attr($featured_width) . esc_attr($featured_unit) . '; height:auto; display:block; margin:0 auto;'))
          . '</div>'
        : '';

    $content = '<div class="nc-carousel-card" style="margin-top:20px;">';
    $content .= '<div class="nc-carousel-content">';
    if ($featured_html) $content .= $featured_html;
    $content .= '<h5 style="text-align:center;">' . esc_html(get_the_title($post)) . '</h5>';
    $content .= '<div style="text-align:center;">' . wp_kses_post(apply_filters('the_content',$post->post_content)) . '</div>';
    $content .= '</div></div>';

    wp_send_json_success($content);
}

/* -------------------------
   Bulk actions
   ------------------------- */
function nc_register_bulk_actions($bulk_actions) {
    $bulk_actions['activate_news'] = 'Activate News';
    $bulk_actions['deactivate_news'] = 'Deactivate News';
    return $bulk_actions;
}
add_filter('bulk_actions-edit-news','nc_register_bulk_actions');

function nc_bulk_action_handler($redirect_to,$doaction,$post_ids) {
    if ($doaction === 'activate_news') {
        foreach ($post_ids as $id) update_post_meta($id,'_nc_news_active','yes');
        $redirect_to = add_query_arg('bulk_activated_news', count($post_ids), $redirect_to);
    } elseif ($doaction === 'deactivate_news') {
        foreach ($post_ids as $id) update_post_meta($id,'_nc_news_active','no');
        $redirect_to = add_query_arg('bulk_deactivated_news', count($post_ids), $redirect_to);
    }
    return $redirect_to;
}
add_filter('handle_bulk_actions-edit-news','nc_bulk_action_handler',10,3);

function nc_bulk_action_notices() {
    if (! empty($_REQUEST['bulk_activated_news'])) printf('<div id="message" class="updated fade">' . esc_html__('Activated %s news items.', 'nc') . '</div>', intval($_REQUEST['bulk_activated_news']));
    if (! empty($_REQUEST['bulk_deactivated_news'])) printf('<div id="message" class="updated fade">' . esc_html__('Deactivated %s news items.', 'nc') . '</div>', intval($_REQUEST['bulk_deactivated_news']));
}
add_action('admin_notices','nc_bulk_action_notices');

add_action('rest_api_init', function () {
    register_rest_route('news/v1', '/create', array(
        'methods'  => 'POST',
        'callback' => 'nc_api_create_news',
        'permission_callback' => function () {
            return current_user_can('edit_posts'); // Adjust capability if needed
        },
        'args' => array(
            'title' => array(
                'required' => true,
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'content' => array(
                'required' => false,
                'sanitize_callback' => 'wp_kses_post',
            ),
            'excerpt' => array(
                'required' => false,
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'active' => array(
                'required' => false,
                'sanitize_callback' => function($v){ return $v === 'yes' ? 'yes' : 'no'; },
            ),
            'start_date' => array(
                'required' => false,
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'end_date' => array(
                'required' => false,
                'sanitize_callback' => 'sanitize_text_field',
            ),
            'show_read_more' => array(
                'required' => false,
                'sanitize_callback' => function($v){ return $v === 'yes' ? 'yes' : 'no'; },
            ),
        )
    ));
});

function nc_api_create_news(WP_REST_Request $request) {
    $title   = $request->get_param('title');
    $content = $request->get_param('content') ?: '';
    $excerpt = $request->get_param('excerpt') ?: '';
    $status  = $request->get_param('status') ?: 'publish';
    $active  = $request->get_param('active') ?: 'no';
    $start   = $request->get_param('start_date') ?: '';
    $end     = $request->get_param('end_date') ?: '';
    $show_rm = $request->get_param('show_read_more') ?: 'yes';

    $post_id = wp_insert_post(array(
        'post_type'    => 'news',
        'post_status'  => $status,   // ← API-driven
        'post_title'   => $title,
        'post_content' => $content,
        'post_excerpt' => $excerpt,
    ));

    if (is_wp_error($post_id)) {
        return new WP_Error('insert_failed', 'Could not create news item.', array('status' => 500));
    }

    // Save meta
    update_post_meta($post_id, '_nc_news_active', $active);
    if ($start) update_post_meta($post_id, '_nc_news_start_date', $start);
    if ($end)   update_post_meta($post_id, '_nc_news_end_date', $end);
    update_post_meta($post_id, '_nc_news_show_read_more', $show_rm);

    return array(
        'success' => true,
        'post_id' => $post_id,
        'status'  => $status,
        'link'    => get_permalink($post_id),
    );
}

/* -------------------------
   Hide "Add Post" submenu under News
   ------------------------- */
function nc_hide_add_new_news_submenu() {
    remove_submenu_page('edit.php?post_type=news', 'post-new.php?post_type=news');
}
add_action('admin_menu', 'nc_hide_add_new_news_submenu', 999);



require_once plugin_dir_path(__FILE__) . 'update.php';
new news_Plugin_Updater(__FILE__, 'news', 'https://it-breeze.cloud/data/news/news.json');
