jQuery(document).ready(function ($) {

  // small helper to build FA icons (unchanged)
  function buildIconHtml(faClass, color) {
    color = color || '#FFD43B';
    var classes = faClass || 'fa-question';
    if (classes.indexOf('fa-') === 0) {
      classes = 'fa-solid ' + classes;
    } else if (classes.indexOf('fa-') > 0 && !classes.includes('fa-solid')) {
      classes = 'fa-solid ' + classes;
    }
    return '<i class="' + classes + '" style="color:' + color + '; margin-right:6px; vertical-align:middle;"></i>';
  }

  function iconFromCode(code) {
    code = Number(code);
    if (isNaN(code)) return buildIconHtml('fa-question', '#888');
    if (code === 0) return buildIconHtml('fa-sun', '#FFD43B');
    if (code >= 1 && code <= 3) return buildIconHtml('fa-cloud-sun', '#FDB813');
    if (code === 45 || code === 48) return buildIconHtml('fa-smog', '#A0A0A0');
    var rain = [51,53,55,56,57,61,63,65,66,67];
    if (rain.indexOf(code) !== -1) return buildIconHtml('fa-cloud-rain', '#1E90FF');
    var snow = [71,73,75,77];
    if (snow.indexOf(code) !== -1) return buildIconHtml('fa-snowflake', '#00BFFF');
    if (code >= 80 && code <= 82) return buildIconHtml('fa-cloud-showers-heavy', '#1E90FF');
    if (code >= 95 && code <= 99) return buildIconHtml('fa-cloud-bolt', '#FFD43B');
    return buildIconHtml('fa-cloud', '#999');
  }

  function parseCsvToArray(csv) {
    if (!csv) return [];
    return String(csv).split(',').map(function (s) { return s.trim(); }).filter(Boolean);
  }

  function asBool(v) {
    return (v === true || v === 1 || v === '1' || String(v).toLowerCase() === 'true');
  }

  function fetchMetarData(icao, container) {
    var globalSettings = (typeof metarWeather !== 'undefined' && metarWeather.settings) ? metarWeather.settings : {};

    var localAttr = {
      header_size: container.attr('data-header-size') || '',
      header_color: container.attr('data-header-color') || '',
      text_size: container.attr('data-text-size') || '',
      text_color: container.attr('data-text-color') || '',
      show_icons: container.attr('data-show-icons'),
      fa_inline_color: container.attr('data-fa-inline-color'),
      compass_size: container.attr('data-compass-size') || '',
      info_to_show: container.attr('data-info-to-show') || ''
    };

    var s = {};
    s.header_size  = localAttr.header_size  || globalSettings.header_size  || '14px';
    s.header_color = localAttr.header_color || globalSettings.header_color || 'inherit';
    s.text_size    = localAttr.text_size    || globalSettings.text_size    || '12px';
    s.text_color   = localAttr.text_color   || globalSettings.text_color   || 'inherit';
    s.show_icons   = (localAttr.show_icons !== '' && typeof localAttr.show_icons !== 'undefined')
                      ? asBool(localAttr.show_icons)
                      : asBool(globalSettings.show_icons);
    s.fa_inline_color = (localAttr.fa_inline_color !== '' && typeof localAttr.fa_inline_color !== 'undefined')
                      ? asBool(localAttr.fa_inline_color)
                      : asBool(globalSettings.fa_inline_color);
    s.compass_size = localAttr.compass_size || globalSettings.compass_size || '22px';
	s.wind_pointer_mode = (globalSettings.wind_pointer_mode || 'from').toLowerCase();

    if (localAttr.info_to_show !== '') {
      s.info_to_show = parseCsvToArray(localAttr.info_to_show);
    } else if (globalSettings.info_to_show) {
      s.info_to_show = Array.isArray(globalSettings.info_to_show)
        ? globalSettings.info_to_show
        : parseCsvToArray(globalSettings.info_to_show);
    } else {
      s.info_to_show = ['sunrise','sunset','temp','humidity','pressure'];
    }

    $.ajax({
      url: metarWeather.ajax_url,
      method: 'GET',
      dataType: 'json',
      data: { action: 'get_metar_weather', icao: icao },
      success: function (response) {
        if (!response.success) {
          container.html('<div class="metar-error">⚠️ ' + (response.data && response.data.message ? response.data.message : 'Unable to fetch data') + '</div>');
          return;
        }

        var weather  = response.data.weather  || {};
        var location = response.data.location || {};
        var sun      = response.data.sun_times || {};

        var iconHtml = '';
        if (s.show_icons) {
          if (weather.icon) {
            var faClass = typeof weather.icon === 'string' ? weather.icon : 'fa-question';
            var color = (s.fa_inline_color && weather.icon_color) ? weather.icon_color : '#FFD43B';
            iconHtml = buildIconHtml(faClass, color);
          } else {
            iconHtml = iconFromCode(weather.weather_code);
          }
        }

        var headerText = location.city ? (location.city + ', ' + (location.country || '')) : (weather.location || icao);

        var html = '<div class="weather" style="font-size:'+s.text_size+'; color:'+s.text_color+';">' +
          '<div class="weather_header" style="font-size:'+s.header_size+'; color:'+s.header_color+'; font-weight:bold;">' +
          iconHtml + headerText + '</div>';

        if (s.info_to_show.indexOf('sunrise') !== -1 && (sun.sunrise_local || sun.sunrise_utc)) {
          html += 'Sunrise: ' + (sun.sunrise_utc || '') + ' UTC / ' + (sun.sunrise_local || '') + ' Local<br>';
        }
        if (s.info_to_show.indexOf('sunset') !== -1 && (sun.sunset_local || sun.sunset_utc)) {
          html += 'Sunset: ' + (sun.sunset_utc || '') + ' UTC / ' + (sun.sunset_local || '') + ' Local<br>';
        }
        if (s.info_to_show.indexOf('temp') !== -1) {
          html += 'Temp: ' + (weather.temp != null ? weather.temp + '°C' : 'N/A') + '<br>';
        }
        if (s.info_to_show.indexOf('humidity') !== -1) {
          html += 'Humidity: ' + (weather.humidity != null ? weather.humidity + '%' : 'N/A') + '<br>';
        }
        if (s.info_to_show.indexOf('pressure') !== -1) {
          html += 'Pressure: ' + (weather.pressure != null ? weather.pressure + ' hPa' : 'N/A') + '<br>';
        }

        // -----------------------------
        // WIND + COMPASS
        // -----------------------------
        if (weather.windspeed !== null && weather.winddirection !== null) {
          var ws = Number(weather.windspeed);
          var wd = Number(weather.winddirection);
          // var rotation = isNaN(wd) ? 0 : ((wd + 180) % 360);
		  //var rotation = isNaN(wd) ? 0 : ((wd) % 360);
		  
	var rotation = isNaN(wd)
  ? 0
  : (s.wind_pointer_mode === 'to'
      ? ((wd + 180) % 360)
      : (wd % 360));
	  



          var info = s.info_to_show || [];
          var windParts = [];
          if (info.indexOf('windspeed') !== -1 && !isNaN(ws)) windParts.push(ws.toFixed(1) + ' km/h');
          if (info.indexOf('winddir')   !== -1 && !isNaN(wd)) windParts.push(wd + '°');
          if (info.indexOf('wind_label')!== -1 && weather.wind_label) windParts.push(weather.wind_label);

          var showCompass = info.indexOf('compass') !== -1;
          if (windParts.length > 0 || showCompass) {
            html += '<div class="wind-info" style="display:inline-flex;align-items:center;gap:4px;">';
            if (windParts.length > 0) {
              html += '<span>Wind: ' + windParts.join(' | ') + '</span>';
            }

            if (showCompass) {
              html += `
                <span class="wind-compass"
                  style="display:inline-block;position:relative;width:${s.compass_size};height:${s.compass_size};margin-left:${windParts.length > 0 ? '6px' : '0'};">
                  <svg viewBox="0 0 200 200" width="100%" height="100%" xmlns="http://www.w3.org/2000/svg">
                    <circle cx="100" cy="100" r="95" fill="#ffffff" stroke="#1e3a8a" stroke-width="10"/>
                    <circle cx="100" cy="100" r="80" fill="#f8fafc" stroke="#94a3b8" stroke-width="2"/>
                    <text x="100" y="25" text-anchor="middle" font-size="16" font-family="Arial" fill="#1e3a8a">N</text>
                    <text x="100" y="185" text-anchor="middle" font-size="16" font-family="Arial" fill="#1e3a8a">S</text>
                    <text x="25" y="105" text-anchor="middle" font-size="16" font-family="Arial" fill="#1e3a8a">W</text>
                    <text x="175" y="105" text-anchor="middle" font-size="16" font-family="Arial" fill="#1e3a8a">E</text>
                    <g transform="translate(100,100) rotate(${rotation})">
                      <polygon points="0,-70 7,0 -7,0" fill="#ef4444"/>
                      <polygon points="0,70 7,0 -7,0" fill="#2563eb"/>
                      <circle cx="0" cy="0" r="5" fill="#1e293b" stroke="#fff" stroke-width="1.5"/>
                    </g>
                  </svg>
                </span>`;
            }

            html += '</div>';
          }
        }

        html += '</div>'; // .weather
        container.html(html);
      },
      error: function (jqXHR, textStatus) {
        container.html('<div class="metar-error">Failed to fetch data (' + textStatus + ').</div>');
      }
    });
  }

  // run on all shortcode containers
  $('.metar-weather-container, .metar-weather').each(function () {
    var container = $(this);
    var icao = container.data('icao') || 'EDDF';
    fetchMetarData(icao, container);
  });

});
