<?php
/**
 * Plugin Name: Weather Display
 * Plugin URI: https://it-breeze.info
 * Update URI: https://it-breeze.cloud/data/weather/weather.json
 * Description: plugin to display weather information.
 * Version: 2.2.0
 * Author: Mike Vahldieck
 * Author URI: https://it-breeze.info
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

function get_country_name_from_iso($iso_code) {
    static $countries = null;

    // Load CSV only once
    if ($countries === null) {
        $countries = [];
        $file = plugin_dir_path(__FILE__) . 'iso-country-codes.csv';

        if (($handle = fopen($file, 'r')) !== false) {
            while (($row = fgetcsv($handle)) !== false) {
                // Skip malformed lines
                if (count($row) < 2) continue;

                $country_name = trim($row[0]);
                $alpha2_code  = trim($row[1]);

                if ($alpha2_code) {
                    $countries[$alpha2_code] = $country_name;
                }
            }
            fclose($handle);
        }
    }

    return $countries[strtoupper($iso_code)] ?? $iso_code;
}


function get_location_from_icao($icao) {
    $stations_file = plugin_dir_path(__FILE__) . 'airports.csv'; // rename if needed

    if (!file_exists($stations_file)) {
        return ['city' => $icao, 'country' => 'Unknown', 'timezone' => 'UTC'];
    }

    $handle = fopen($stations_file, 'r');
    if (!$handle) {
        return ['city' => $icao, 'country' => 'Unknown', 'timezone' => 'UTC'];
    }

    while (($parts = fgetcsv($handle)) !== false) {
        // Skip if line is incomplete or malformed
        if (count($parts) < 11) {
            continue;
        }

        // Column 1 = ICAO
        if (trim($parts[1]) === strtoupper($icao)) {
            fclose($handle);

            $city = trim($parts[10]);
            $iso_code = trim($parts[8]);
            $country = get_country_name_from_iso($iso_code);
            $timezone = get_timezone_from_country($iso_code); // optionally keep using ISO code for timezone

            
            return [
                'city'     => $city ?: $icao,
                'country'  => $country ?: 'Unknown',
                'timezone' => $timezone ?: 'UTC'
            ];
        }
    }

    fclose($handle);
    return ['city' => $icao, 'country' => 'Unknown', 'timezone' => 'UTC'];
}

function get_timezone_from_country($iso_country) {
    // Use PHP’s built-in mapping
    $timezones = DateTimeZone::listIdentifiers(DateTimeZone::PER_COUNTRY, $iso_country);
    if (!empty($timezones)) {
        return $timezones[0]; // use the first (main) timezone for that country
    }
    return 'UTC';
}



// Function to get latitude and longitude using Nominatim API
function get_lat_lon($city) {
    $nominatim_url = 'https://nominatim.openstreetmap.org/search?q=' . urlencode($city) . '&format=json&limit=1';
    $ch = curl_init();

    curl_setopt($ch, CURLOPT_URL, $nominatim_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'WeatherDisplayPlugin/2.2.0 (contact: info@it-breeze.info)');



    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    curl_close($ch);

    if ($http_code !== 200 || !$response) {
        return ['lat' => null, 'lon' => null];
    }

    $data = json_decode($response, true);
    if (isset($data[0]['lat']) && isset($data[0]['lon'])) {
        return [
            'lat' => floatval($data[0]['lat']),
            'lon' => floatval($data[0]['lon'])
        ];
    }

    return ['lat' => null, 'lon' => null];
}

// ==============================
// Sunrise/sunset calculation (defensive)
// ==============================
function calculate_sun_times($latitude, $longitude, $timezone) {
    $timestamp = time();

    if (empty($timezone)) {
        $timezone = 'UTC';
    }

    $sun_info = date_sun_info($timestamp, $latitude, $longitude);

    if (!isset($sun_info['sunrise']) || !isset($sun_info['sunset'])) {
        return [
            'sunrise_utc' => '',
            'sunset_utc' => '',
            'sunrise_local' => '',
            'sunset_local' => ''
        ];
    }

    try {
        $sunrise_dt_utc = (new DateTime())->setTimestamp($sun_info['sunrise'])->setTimezone(new DateTimeZone('UTC'));
        $sunset_dt_utc  = (new DateTime())->setTimestamp($sun_info['sunset'])->setTimezone(new DateTimeZone('UTC'));

        $local_tz = new DateTimeZone($timezone);
        $sunrise_local = clone $sunrise_dt_utc;
        $sunrise_local->setTimezone($local_tz);
        $sunset_local = clone $sunset_dt_utc;
        $sunset_local->setTimezone($local_tz);

        return [
            'sunrise_utc'   => $sunrise_dt_utc->format('H:i'),
            'sunset_utc'    => $sunset_dt_utc->format('H:i'),
            'sunrise_local' => $sunrise_local->format('H:i'),
            'sunset_local'  => $sunset_local->format('H:i'),
        ];
    } catch (Exception $e) {
        // fallback to UTC strings
        return [
            'sunrise_utc'   => gmdate('H:i', $sun_info['sunrise']),
            'sunset_utc'    => gmdate('H:i', $sun_info['sunset']),
            'sunrise_local' => gmdate('H:i', $sun_info['sunrise']),
            'sunset_local'  => gmdate('H:i', $sun_info['sunset']),
        ];
    }
}

// ==============================
// Shortcode (preserve your original HTML)
// ==============================
function metar_weather_shortcode($atts) {
    // Keep raw user attributes (to detect what was explicitly provided)
    $user_atts = $atts;

    // Admin defaults
    $opts = metar_get_settings();

    // Default shortcode attributes (pull defaults from admin options)
    $defaults = [
        'icao'            => '',
        'header_size'     => $opts['header_size'],
        'header_color'    => $opts['header_color'],
        'text_size'       => $opts['text_size'],
        'text_color'      => $opts['text_color'],
        'show_icons'      => $opts['show_icons'],
        'fa_inline_color' => $opts['fa_inline_color'],
        'compass_size'    => isset($opts['compass_size']) ? $opts['compass_size'] : '22px',
        // allow full CSV override if desired
        'info_to_show'    => is_array($opts['info_to_show']) ? implode(',', $opts['info_to_show']) : (isset($opts['info_to_show']) ? $opts['info_to_show'] : ''),
		'wind_pointer_mode' => 'from',
    ];
    

    // Merge with user attributes
    $atts = shortcode_atts($defaults, $atts, 'metar_weather');

    // Normalize some booleans
    $atts['show_icons'] = ($atts['show_icons'] === '1' || strtolower((string)$atts['show_icons']) === 'true') ? '1' : '0';
    $atts['fa_inline_color'] = ($atts['fa_inline_color'] === '1' || strtolower((string)$atts['fa_inline_color']) === 'true') ? '1' : '0';

    // Determine which info fields to show.
    // Accept both individual flags (sunrise="1") or show_sunrise="1", OR info_to_show CSV.
    $fields = ['sunrise','sunset','temp','humidity','pressure','windspeed','winddir','wind_label','compass'];

    $info = [];
    $provided_individual = false;

    // If user passed explicit individual attributes, use them (highest precedence)
    foreach ($fields as $f) {
        if (array_key_exists($f, $user_atts) || array_key_exists('show_' . $f, $user_atts)) {
            $provided_individual = true;
            $val = null;
            if (array_key_exists($f, $user_atts)) {
                $val = $user_atts[$f];
            } else {
                $val = $user_atts['show_' . $f];
            }
            // treat '1', 'true', 'yes', 'on' as true
            $val_str = strtolower(trim((string)$val));
            if (in_array($val_str, ['1','true','yes','on'], true)) {
                $info[] = $f;
            }
        }
    }

    // If no individual flags were provided, check CSV info_to_show (user attr) first,
    // otherwise use merged $atts['info_to_show'] (which already falls back to admin).
    if (!$provided_individual) {
        $csv = '';
        if (!empty($user_atts['info_to_show'])) {
            $csv = $user_atts['info_to_show'];
        } elseif (!empty($atts['info_to_show'])) {
            $csv = $atts['info_to_show'];
        }

        if (!empty($csv)) {
            $parts = array_map('trim', explode(',', $csv));
            foreach ($parts as $p) {
                if ($p !== '' && in_array($p, $fields, true)) {
                    $info[] = $p;
                }
            }
        }
    }

    // Final fallback to admin defaults if still empty
    if (empty($info)) {
        $info = is_array($opts['info_to_show']) ? $opts['info_to_show'] : ( !empty($opts['info_to_show']) ? explode(',', $opts['info_to_show']) : [] );
        $info = array_values(array_filter(array_map('trim', $info)));
    }

    // Ensure unique and valid field names
    $info = array_values(array_unique(array_intersect($fields, $info)));

    // Build final attributes for data-* output
    $icao = sanitize_text_field($atts['icao']);
    if (empty($icao)) {
        return '<p><strong>Error:</strong> Missing ICAO code.</p>';
    }

    // sanitize other attributes for output
    $header_size   = sanitize_text_field($atts['header_size']);
    $header_color  = sanitize_text_field($atts['header_color']);
    $text_size     = sanitize_text_field($atts['text_size']);
    $text_color    = sanitize_text_field($atts['text_color']);
    $show_icons    = $atts['show_icons']; // already normalized '1' or '0'
    $fa_inline     = $atts['fa_inline_color']; // '1' or '0'
    $compass_size  = sanitize_text_field($atts['compass_size']);

    // Output container with data-* attributes
    ob_start();
    ?>
    <div class="metar-weather-container"
         data-icao="<?php echo esc_attr($icao); ?>"
         data-header-size="<?php echo esc_attr($header_size); ?>"
         data-header-color="<?php echo esc_attr($header_color); ?>"
         data-text-size="<?php echo esc_attr($text_size); ?>"
         data-text-color="<?php echo esc_attr($text_color); ?>"
         data-show-icons="<?php echo esc_attr($show_icons); ?>"
         data-fa-inline-color="<?php echo esc_attr($fa_inline); ?>"
         data-info-to-show="<?php echo esc_attr(implode(',', $info)); ?>"
         data-compass-size="<?php echo esc_attr($compass_size); ?>">
        <div class="metar-loading">
            <img src="<?php echo plugin_dir_url(__FILE__) . 'running.gif'; ?>" alt="Loading..." />
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('metar_weather', 'metar_weather_shortcode');
add_shortcode('weather', 'metar_weather_shortcode'); // alias

function metar_weather_enqueue_scripts() {
    wp_enqueue_style('metar-fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css', [], '6.5.0');
    wp_enqueue_script('jquery');

    wp_enqueue_script(
        'metar-weather-script',
        plugin_dir_url(__FILE__) . 'weather.js',
        ['jquery'],
        '1.4',
        true
    );

    $settings = get_option('metar_weather_settings', []);
    wp_localize_script('metar-weather-script', 'metarWeather', [
        'ajax_url'   => admin_url('admin-ajax.php'),
        'plugin_url' => plugin_dir_url(__FILE__),
        'settings'   => $settings,
    ]);
}
add_action('wp_enqueue_scripts', 'metar_weather_enqueue_scripts');

function metar_weather_enqueue_styles() {
    wp_enqueue_style('metar-weather-style', plugin_dir_url(__FILE__) . 'weather.css');
}
add_action('wp_enqueue_scripts', 'metar_weather_enqueue_styles');

// ==============================
// AJAX handler: get_metar_weather (uses Open-Meteo)
// ==============================
add_action('wp_ajax_get_metar_weather', 'metar_weather_ajax_handler');
add_action('wp_ajax_nopriv_get_metar_weather', 'metar_weather_ajax_handler');

// -----------------------
// Debugged AJAX handler (replace your existing handler body with this)
// -----------------------
function metar_weather_ajax_handler() {
    $icao = isset($_GET['icao']) ? sanitize_text_field($_GET['icao']) : 'EDDF';
    $icao = strtoupper($icao);

    // 1) Get station info
    $location = get_location_from_icao($icao);
    if (!$location || empty($location['city'])) {
        wp_send_json_error(['message' => 'Unknown ICAO code: ' . esc_html($icao)]);
        return;
    }

	$geo = get_lat_lon($location['city'] . ', ' . $location['country']);
    if ($geo['lat'] === null || $geo['lon'] === null) {
        wp_send_json_error(['message' => 'Unable to fetch geographic data for ' . esc_html($location['city'])]);
        return;
    }
    $lat = $geo['lat'];
    $lon = $geo['lon'];

    // 3) Request Open-Meteo (current + hourly humidity/pressure)
    $open_url = sprintf(
        'https://api.open-meteo.com/v1/forecast?latitude=%s&longitude=%s&current_weather=true&hourly=relativehumidity_2m,pressure_msl&timezone=UTC',
        rawurlencode($lat),
        rawurlencode($lon)
    );

    $response = wp_remote_get($open_url, [
        'timeout' => 10,
        'headers' => [
            'User-Agent' => 'WP weather plugin/1.0 (info@it-breeze.info)'
        ],
    ]);

    if (is_wp_error($response)) {
        wp_send_json_error(['message' => 'Failed to fetch weather data: ' . $response->get_error_message()]);
        return;
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);


    if (empty($data) || !isset($data['current_weather'])) {
        wp_send_json_error(['message' => 'No current weather data available from Open-Meteo.']);
        return;
    }

    $current = $data['current_weather']; // temperature, weathercode, time
    $windspeed    = isset($current['windspeed'])     ? floatval($current['windspeed'])     : null;
    $winddir      = isset($current['winddirection']) ? intval($current['winddirection'])   : null;
    function metar_wind_direction_label($deg) {
        if ($deg === null) return null;
        $dirs = ['N','NE','E','SE','S','SW','W','NW'];
        return $dirs[(int)(($deg + 22.5) / 45) % 8];
    }
    $wind_label = metar_wind_direction_label($winddir);
    // find matching hourly humidity/pressure (nearest timestamp)
    $hourly_times = $data['hourly']['time'] ?? null;
    $hourly_humidity = $data['hourly']['relativehumidity_2m'] ?? null;
    $hourly_pressure = $data['hourly']['pressure_msl'] ?? null;

    $humidity = null; $pressure = null;
    $current_time = $current['time'] ?? null;
    if ($hourly_times && $current_time) {
        $idx = array_search($current_time, $hourly_times);
        if ($idx === false) {
            // nearest index
            $closest = null; $closestDiff = PHP_INT_MAX;
            foreach ($hourly_times as $i => $t) {
                $diff = abs(strtotime($t) - strtotime($current_time));
                if ($diff < $closestDiff) { $closestDiff = $diff; $closest = $i; }
            }
            $idx = $closest;
        }
        if ($idx !== null && isset($hourly_humidity[$idx])) $humidity = $hourly_humidity[$idx];
        if ($idx !== null && isset($hourly_pressure[$idx])) $pressure = $hourly_pressure[$idx];
    }

    // map weather code -> icon + color (your mapping)
    $weather_code = isset($current['weathercode']) ? (int)$current['weathercode'] : null;
    $icon = 'fa-question'; $icon_color = '#888888';
    if ($weather_code === 0) {
        $icon = 'fa-sun'; $icon_color = '#FFD43B';
    } elseif ($weather_code >= 1 && $weather_code <= 3) {
        $icon = 'fa-cloud-sun'; $icon_color = '#FDB813';
    } elseif (in_array($weather_code, [45,48])) {
        $icon = 'fa-smog'; $icon_color = '#A0A0A0';
    } elseif (in_array($weather_code, [51,53,55,56,57,61,63,65,66,67])) {
        $icon = 'fa-cloud-rain'; $icon_color = '#1E90FF';
    } elseif (in_array($weather_code, [71,73,75,77])) {
        $icon = 'fa-snowflake'; $icon_color = '#00BFFF';
    } elseif ($weather_code >= 80 && $weather_code <= 82) {
        $icon = 'fa-cloud-showers-heavy'; $icon_color = '#1E90FF';
    } elseif ($weather_code >= 95 && $weather_code <= 99) {
        $icon = 'fa-cloud-bolt'; $icon_color = '#FFD43B';
    } else {
        $icon = 'fa-cloud'; $icon_color = '#999999';
    }

    $weather = [
        'temp' => isset($current['temperature']) ? round($current['temperature'], 1) : null,
        'humidity' => $humidity !== null ? round($humidity, 1) : null,
        'pressure' => $pressure !== null ? round($pressure, 1) : null,
        'windspeed'     => $windspeed,
        'winddirection' => $winddir,
        'wind_label'    => $wind_label,
        'weather_code' => $weather_code,
        'icon' => $icon,
        'icon_color' => $icon_color
    ];

    $tz = !empty($location['timezone']) ? $location['timezone'] : 'UTC';
    $sun_times = calculate_sun_times($lat, $lon, $tz);

    // ===== DEBUG: include raw API parsed array in the JSON reply (temporary) =====
    wp_send_json_success([
        'weather' => $weather,
        'location' => $location,
        'sun_times' => $sun_times,
        'raw_api' => $data,              // remove this line after debugging
    ]);
    // =================================================================================
}

// ---------------------------
// Admin: Settings page (Weather)
// ---------------------------

// Register admin menu + settings
add_action('admin_menu', 'metar_weather_add_admin_menu');
add_action('admin_init', 'metar_weather_settings_init');
add_action('admin_enqueue_scripts', 'metar_weather_admin_assets');

/**
 * Add menu under Settings
 */
function metar_weather_add_admin_menu() {
    add_options_page(
        'Weather Display Settings', // Page title
        'Weather',                  // Menu label (Settings -> Weather)
        'manage_options',
        'metar_weather',
        'metar_weather_options_page'
    );
}

/**
 * Enqueue WP color picker only on our settings page
 */
function metar_weather_admin_assets($hook_suffix) {
    // Only load when on our settings page
    if (isset($_GET['page']) && $_GET['page'] === 'metar_weather') {
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_script('jquery');
        // optional small inline script to init color pickers will be printed in options page
    }
}

/**
 * Register settings, sections and fields
 */
function metar_weather_settings_init() {
    register_setting('metar_weather_group', 'metar_weather_settings', 'metar_weather_sanitize');

    add_settings_section(
        'metar_weather_section_main',
        'Weather display settings',
        function() { echo '<p>Configure plugin defaults. Leave blank to use theme defaults where applicable.</p>'; },
        'metar_weather'
    );

    // Header font size
    add_settings_field(
        'metar_header_size',
        'Header size',
        'metar_field_header_size_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Header color
    add_settings_field(
        'metar_header_color',
        'Header color',
        'metar_field_header_color_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Text size
    add_settings_field(
        'metar_text_size',
        'Text size',
        'metar_field_text_size_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Text color
    add_settings_field(
        'metar_text_color',
        'Text color',
        'metar_field_text_color_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Show icons
    add_settings_field(
        'metar_show_icons',
        'Show icons',
        'metar_field_show_icons_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Use FA inline color
    add_settings_field(
        'metar_fa_inline_color',
        'Use Font Awesome inline icon color',
        'metar_field_fa_inline_color_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    // Which information to show
    add_settings_field(
        'metar_info_to_show',
        'Information to display',
        'metar_field_info_to_show_cb',
        'metar_weather',
        'metar_weather_section_main'
    );

    add_settings_field(
        'metar_compass_size',
        'Compass size',
        'metar_field_compass_size_cb',
        'metar_weather',
        'metar_weather_section_main'
    );
	
    add_settings_field(
        'metar_wind_pointer_mode',
        'Wind pointer mode',
        'metar_field_wind_pointer_mode_cb',
        'metar_weather',
        'metar_weather_section_main'
    );
	

	register_setting('metar_weather_options_group', 'metar_weather_wind_pointer_mode', [
		'type' => 'string',
		'sanitize_callback' => function($value) {
			return in_array(strtolower($value), ['from', 'to'], true) ? strtolower($value) : 'from';
		},
		'default' => 'from',
	]);

}

/* ---------------------------
   Field callbacks
   --------------------------- */

function metar_get_settings() {
    $defaults = [
        'header_size'   => '',
        'header_color'  => '',
        'text_size'     => '',
        'text_color'    => '',
        'show_icons'    => '1',
        'fa_inline_color'=> '1',
        'info_to_show'  => ['sunrise','sunset','temp','humidity','pressure','windspeed','winddir','wind_label','compass'],
        'compass_size'  => '22px',
    ];
    $opts = get_option('metar_weather_settings', []);
    return wp_parse_args($opts, $defaults);
}

function metar_field_header_size_cb() {
    $opts = metar_get_settings();
    ?>
    <input type="text" name="metar_weather_settings[header_size]" value="<?php echo esc_attr($opts['header_size']); ?>" placeholder="e.g. 14px or 1.2rem">
    <p class="description">Header font size (leave empty to use theme default).</p>
    <?php
}

function metar_field_header_color_cb() {
    $opts = metar_get_settings();
    ?>
    <input class="metar-color-field" type="text" name="metar_weather_settings[header_color]" value="<?php echo esc_attr($opts['header_color']); ?>" placeholder="#000000">
    <p class="description">Hex color for header (leave empty to inherit theme color).</p>
    <?php
}

function metar_field_text_size_cb() {
    $opts = metar_get_settings();
    ?>
    <input type="text" name="metar_weather_settings[text_size]" value="<?php echo esc_attr($opts['text_size']); ?>" placeholder="e.g. 12px">
    <p class="description">Main text size (leave empty to use theme default).</p>
    <?php
}

function metar_field_text_color_cb() {
    $opts = metar_get_settings();
    ?>
    <input class="metar-color-field" type="text" name="metar_weather_settings[text_color]" value="<?php echo esc_attr($opts['text_color']); ?>" placeholder="#333333">
    <p class="description">Main text color (leave empty to inherit theme color).</p>
    <?php
}

function metar_field_show_icons_cb() {
    $opts = metar_get_settings();
    $checked = ($opts['show_icons'] == '1') ? 'checked' : '';
    ?>
    <label><input type="checkbox" name="metar_weather_settings[show_icons]" value="1" <?php echo $checked; ?>> Show icons</label>
    <p class="description">Enable/disable weather icons globally.</p>
    <?php
}

function metar_field_fa_inline_color_cb() {
    $opts = metar_get_settings();
    $checked = ($opts['fa_inline_color'] == '1') ? 'checked' : '';
    ?>
    <label><input type="checkbox" name="metar_weather_settings[fa_inline_color]" value="1" <?php echo $checked; ?>> Allow Font Awesome icons to use inline color (preferred)</label>
    <p class="description">If unchecked, icons will use theme CSS color instead of inline color.</p>
    <?php
}

function metar_field_compass_size_cb() {
    $opts = metar_get_settings();
    ?>
    <input type="text" name="metar_weather_settings[compass_size]" value="<?php echo esc_attr($opts['compass_size']); ?>" placeholder="e.g. 22px or 1.4em">
    <p class="description">Set the size of the compass icon (default 22px).</p>
    <?php
}   

function metar_field_info_to_show_cb() {
    $opts = metar_get_settings();
    $selected = is_array($opts['info_to_show']) ? $opts['info_to_show'] : [];
    $items = [
        'sunrise'    => 'Sunrise (UTC / Local)',
        'sunset'     => 'Sunset (UTC / Local)',
        'temp'       => 'Temp',
        'humidity'   => 'Humidity',
        'pressure'   => 'Pressure',
        'windspeed'  => 'Windspeed',
        'winddir'    => 'Wind direction (°)',
        'wind_label' => 'Wind direction text',
        'compass'    => 'Compass icon 🧭'
    ];
    foreach ($items as $key => $label) {
        $chk = in_array($key, $selected) ? 'checked' : '';
        echo '<label style="display:block;"><input type="checkbox" name="metar_weather_settings[info_to_show][]" value="' . esc_attr($key) . '" ' . $chk . '> ' . esc_html($label) . '</label>';
    }
    echo '<p class="description">Choose which lines are shown in the widget.</p>';
}

function metar_field_wind_pointer_mode_cb() {
    $opts = metar_get_settings();
    $current = isset($opts['wind_pointer_mode']) ? $opts['wind_pointer_mode'] : 'from';
    ?>
    <select name="metar_weather_settings[wind_pointer_mode]">
        <option value="from" <?php selected($current, 'from'); ?>>Needle points where wind is coming FROM (meteorological standard)</option>
        <option value="to" <?php selected($current, 'to'); ?>>Needle points where wind is blowing TO</option>
    </select>
    <p class="description">Choose how the compass needle should indicate wind direction.</p>
    <?php
}


/* ---------------------------
   Sanitize callback
   --------------------------- */
function metar_weather_sanitize($input) {
    $clean = [];

    $clean['header_size'] = isset($input['header_size']) ? sanitize_text_field($input['header_size']) : '';
    $clean['text_size']   = isset($input['text_size']) ? sanitize_text_field($input['text_size']) : '';

    $clean['header_color'] = isset($input['header_color']) ? sanitize_hex_color($input['header_color']) : '';
    $clean['text_color']   = isset($input['text_color']) ? sanitize_hex_color($input['text_color']) : '';

    $clean['show_icons'] = (isset($input['show_icons']) && $input['show_icons']) ? '1' : '0';
    $clean['fa_inline_color'] = (isset($input['fa_inline_color']) && $input['fa_inline_color']) ? '1' : '0';

    // NEW: compass_size
    $clean['compass_size'] = isset($input['compass_size']) ? sanitize_text_field($input['compass_size']) : '22px';

    $allowed_info = ['sunrise','sunset','temp','humidity','pressure','windspeed','winddir','wind_label','compass'];
    $clean['info_to_show'] = [];
    if (!empty($input['info_to_show']) && is_array($input['info_to_show'])) {
        foreach ($input['info_to_show'] as $val) {
            if (in_array($val, $allowed_info)) {
                $clean['info_to_show'][] = $val;
            }
        }
    }
    if (empty($clean['info_to_show'])) {
        $clean['info_to_show'] = $allowed_info;
    }
	
	if (isset($input['wind_pointer_mode'])) {
        $val = strtolower(trim($input['wind_pointer_mode']));
        $clean['wind_pointer_mode'] = in_array($val, ['from', 'to'], true) ? $val : 'from';
    } else {
        $clean['wind_pointer_mode'] = 'from';
    }

    return $clean;
}

/* ---------------------------
   Render the options page
   --------------------------- */
function metar_weather_options_page() {
    if (!current_user_can('manage_options')) {
        return;
    }
    ?>
    <div class="wrap">
        <h1>Weather Display Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('metar_weather_group');
            do_settings_sections('metar_weather');
            submit_button();
            ?>
        </form>
    </div>

    <script type="text/javascript">
    jQuery(document).ready(function($){
        // Initialize WP color picker on our color inputs (only present on this settings page)
        if ($.fn.wpColorPicker) {
            $('.metar-color-field').wpColorPicker({
                defaultColor: '',
                hide: true,
                palettes: true
            });
        }
    });
    </script>
    <?php
}

require_once plugin_dir_path(__FILE__) . 'update.php';
new weather_Plugin_Updater(__FILE__, 'weather', 'https://it-breeze.cloud/data/weather/weather.json');